/* Plagiat gallery creation tool for Jakub Steiner's O.R.I.G.I.N.A.L 
 * photo gallery.
 * Copyright (C) 2005 Robert Staudinger <robert.staudinger@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <stdio.h>
#include <gsf/gsf-utils.h>
#include <gsf/gsf-input-stdio.h>
#include <gsf/gsf-input-textline.h>
#include <gsf/gsf-outfile.h>
#include <gsf/gsf-output-stdio.h>
#include <gsf/gsf-outfile-stdio.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gtk/gtk.h>
#include "pgt-controller.h"
#include "pgt-utils.h"

struct _PgtController {
	GObject parent;
	gboolean is_disposed;
	PgtModel *model;
	PgtView *view;
};

struct _PgtControllerClass {
	GObjectClass parent;
};

/* properties */
enum {
	_PROP_0,
	PROP_MODEL,
	PROP_VIEW,
	_NUM_PROPS
};

static GObjectClass *pgt_controller_parent_class = NULL;


static void 
add_gallery_cb (PgtController *self, 
		GtkToolButton *button)
{
	gchar const *name = NULL;
	gchar const *url = NULL;
	gboolean ret;

	name = pgt_view_gallery_get_name (self->view);
	url = pgt_view_gallery_get_url (self->view);

	ret = pgt_model_add_gallery (self->model, name, url);
	
	/* TODO error
	if (!ret) {
		GtkWidget *errordlg = NULL;
		GtkWidget *parent = pgt_galleries_get_window (self->galleries);
		errordlg = gtk_message_dialog_new (GTK_WINDOW (parent), GTK_DIALOG_MODAL,
						   GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
						   "Name und URL müssen angegeben werden");
		gtk_dialog_run (GTK_DIALOG (errordlg));
		gtk_widget_destroy (errordlg);
	}
	*/
}

static void 
del_gallery_cb (PgtController *self,
		GtkToolButton *button)
{
	gchar *name = NULL;
	gboolean ret;
	
	name = pgt_view_gallery_get_selected_name (self->view);

	ret = pgt_model_del_gallery (self->model, name);
	if (!ret) {
		/* TODO error */
	}

	if (name) {
		g_free (name);
	}
}

static void 
process_cb (PgtController *self, 
	    PgtView       *view)
{
	pgt_controller_convert (self);
}

static void
instance_init (PgtController *self)
{
	self->is_disposed = FALSE;
	self->model = NULL;
	self->view = NULL;
}

static void
instance_dispose (GObject *instance)
{
	PgtController *self = PGT_CONTROLLER (instance);

	if (self->is_disposed)
		return;

	g_object_unref (self->view);
	self->is_disposed = TRUE;

	if (self->model) {
		g_object_unref (self->model);
		self->model = NULL;
	}
	if (self->view) {
		g_object_unref (self->view);
		self->view = NULL;
	}

	pgt_controller_parent_class->dispose (G_OBJECT (self));
}

static void
set_property (GObject      *object,
	      guint         prop_id,
	      GValue const *value,
	      GParamSpec   *pspec)
{
	PgtController *self = PGT_CONTROLLER (object);
	gpointer *p = NULL;

	switch (prop_id) {
	case PROP_MODEL:
		p = g_value_get_object (value);
		if (p) {
			self->model = PGT_MODEL (p);
			g_object_ref (self->model);
		}
		else if (self->model) {
			g_object_unref (self->model);
			self->model = NULL;
		}
		break;
	case PROP_VIEW:
		p = g_value_get_object (value);
		if (p) {
			self->view = PGT_VIEW (p);
			g_object_ref (self->view);

			g_signal_connect_swapped (self->view, "add-gallery", 
						  G_CALLBACK (add_gallery_cb), self);
			g_signal_connect_swapped (self->view, "del-gallery", 
						  G_CALLBACK (del_gallery_cb), self);
			g_signal_connect_swapped (self->view, "process", 
						  G_CALLBACK (process_cb), self);

		}
		else if (self->view) {
			g_object_unref (self->view);
			self->view = NULL;
		}
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
	}
}

static void
get_property (GObject    *object,
	      guint       prop_id,
	      GValue     *value,
	      GParamSpec *pspec)
{
	PgtController *self = PGT_CONTROLLER (object);

	switch (prop_id) {
	case PROP_MODEL:
		g_value_set_pointer (value, (gpointer) self->model);
		break;
	case PROP_VIEW:
		g_value_set_pointer (value, (gpointer) self->view);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
	}
}

static void
class_init (PgtControllerClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

	/* hook gobject vfuncs */
	gobject_class->dispose = instance_dispose;

	gobject_class->set_property = set_property;
	gobject_class->get_property = get_property;

	pgt_controller_parent_class = (GObjectClass*) g_type_class_peek_parent (klass);

	g_object_class_install_property (gobject_class,
		PROP_MODEL,
		g_param_spec_object ("model", "Model Pointer",
			"Pointer to model instance", PGT_TYPE_MODEL,
			(GParamFlags)(G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY)));

	g_object_class_install_property (gobject_class,
		PROP_VIEW,
		g_param_spec_object ("view", "View Pointer",
			"Pointer to view instance", PGT_TYPE_VIEW,
			(GParamFlags)(G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY)));
}

GType
pgt_controller_get_gtype (void)
{
        static GType type = 0;
        if (!type) {
                static const GTypeInfo info = {
                        sizeof (PgtControllerClass),
                        NULL,           /* base_init */
                        NULL,           /* base_finalize */
                        (GClassInitFunc) class_init,
                        NULL,           /* class_finalize */
                        NULL,           /* class_data */
                        sizeof (PgtController),
                        0,              /* n_preallocs */
                        (GInstanceInitFunc) instance_init,
                };
                type = g_type_register_static (G_TYPE_OBJECT, "PgtController", &info, (GTypeFlags)0);
        }
        return type;
}

PgtController*   
pgt_controller_new (PgtModel *model, PgtView *view)
{
	return PGT_CONTROLLER (g_object_new (PGT_TYPE_CONTROLLER, 
					     "model", model, 
					     "view", view, 
					     NULL));
}

static gboolean
is_img (const gchar *filename)
{
	#define NUM_PATTERNS 4
	static GPatternSpec *patterns[NUM_PATTERNS];
	static gboolean patterns_compiled = FALSE;
	int i;

	if (!patterns_compiled) {
		patterns[0] = g_pattern_spec_new ("*.jpeg");
		patterns[1] = g_pattern_spec_new ("*.JPEG");
		patterns[2] = g_pattern_spec_new ("*.jpg");
		patterns[3] = g_pattern_spec_new ("*.JPG");
		patterns_compiled = TRUE;
	}

	for (i = 0; i < NUM_PATTERNS; i++) {
		if (g_pattern_match_string (patterns[i], filename)) {
			return TRUE;
		}
	}
	return FALSE;

	#undef NUM_PATTERNS
}

static gboolean
convert (const gchar *srcname,
	 GsfOutfile  *dstdir,  
	 const gchar *dstname, 
	 guint 	      dst_width, 
	 guint        dst_height, 
	 GdkInterpType quality)
{
	GdkPixbuf *src = NULL;
	GdkPixbuf *dst = NULL;
	GsfOutput *ofile = NULL;
	GError *err = NULL;
        int src_width;
        int src_height;
	gdouble ratio;
	gchar *buffer = NULL;
	gsize bufsize;

	src = gdk_pixbuf_new_from_file (srcname, &err);
	if (err) {
		g_error (err->message);
	}

        src_width = gdk_pixbuf_get_width (src);
        src_height = gdk_pixbuf_get_height (src);

	if (src_width > src_height) {
		ratio = (gdouble)src_width / (gdouble)dst_width;
		dst_height = src_height / ratio;
	}
	else {
		ratio = (gdouble)src_height / (gdouble)dst_height;
		dst_width = src_width / ratio;
	}

	dst = gdk_pixbuf_scale_simple (src, dst_width, dst_height, quality);
	ofile = gsf_outfile_new_child (dstdir, dstname, FALSE);
	
	gdk_pixbuf_save_to_buffer (dst, &buffer, &bufsize, "jpeg", &err, "quality", "100", NULL);
	if (err) {
		g_error (err->message);
	}
	gsf_output_write (ofile, bufsize, (guint8 *) buffer);
	
	g_free (buffer);
	buffer = NULL;
	gsf_output_close (ofile);

	return TRUE;
}

static void
comment (GsfOutfile *dstdir,
	 int 	     num)

{
	GsfOutput *ofile = NULL;
	gchar *dstname = NULL;
	
	dstname = g_strdup_printf ("%d.txt", num);
	ofile = gsf_outfile_new_child (dstdir, dstname, FALSE);
	g_free (dstname);	

	gsf_output_printf (ofile, _("<span>Photo %d</span>"), num);
	gsf_output_close (ofile);
}

static guint
count_imgs (GDir *idir)
{
	const gchar *filename = NULL;
	guint count = 0;

	while (NULL != (filename = g_dir_read_name (idir))) {

		if (is_img (filename)) {
			count++;
		}
	}

	g_dir_rewind (idir);
	return count;
}

static void 
create_gallery (PgtController	      *self,
		const gchar 	      *basedir,
		GDir		      *idir,
		guint		       num_imgs,
		GsfOutfile 	      *odir, 
		gboolean  	       options[num_options], 
		GdkInterpType 	       quality)
{
	GsfOutput *thumbdir = NULL;
	GsfOutput *lqdir = NULL;
	GsfOutput *mqdir = NULL;
	GsfOutput *hqdir = NULL;
	GsfOutput *commentdir = NULL;
	GsfOutput *zipdir = NULL;
	const gchar *filename = NULL;
	gchar *srcname = NULL;
	gchar *dstname = NULL;
	int i = 0;
	gfloat progress = 0;


	thumbdir = gsf_outfile_new_child (odir, "thumbs", TRUE);
	if (options[option_lq])
		lqdir = gsf_outfile_new_child (odir, "lq", TRUE);
	if (options[option_mq])
		mqdir = gsf_outfile_new_child (odir, "mq", TRUE);
	if (options[option_hq])
		hqdir = gsf_outfile_new_child (odir, "hq", TRUE);
	if (options[option_comment])
		commentdir = gsf_outfile_new_child (odir, "comments", TRUE);
	if (options[option_zip])
		zipdir = gsf_outfile_new_child (odir, "zip", TRUE);


	while (NULL != (filename = g_dir_read_name (idir))) {

		if (is_img (filename)) {

			i++;
			srcname = g_build_path (G_DIR_SEPARATOR_S, basedir, filename, NULL);
			dstname = g_strdup_printf ("img-%d.jpg", i);

			if (thumbdir) 
				convert (srcname, GSF_OUTFILE (thumbdir), dstname, 120, 120, quality);
			if (lqdir) 
				convert (srcname, GSF_OUTFILE (lqdir), dstname, 640, 480, quality);
			if (mqdir) 
				convert (srcname, GSF_OUTFILE (mqdir), dstname, 1024, 768, quality);
			if (hqdir) 
				/* TODO 
				copy */ ;
			if (commentdir) 
				comment (GSF_OUTFILE (commentdir), i);
			if (zipdir) 
				/* TODO */ ;

			g_free (srcname);
			srcname = NULL;
			g_free (dstname);
			dstname = NULL;

			progress = (gfloat)i / (gfloat)num_imgs;
			pgt_view_set_progress_value (self->view, progress);
			gtk_main_iteration ();
		}
	}


	if (thumbdir) 
		gsf_output_close (thumbdir);
	if (lqdir)
		gsf_output_close (lqdir);
	if (mqdir)
		gsf_output_close (mqdir);
	if (hqdir)
		gsf_output_close (hqdir);
	if (commentdir)
		gsf_output_close (commentdir);
	if (zipdir)
		gsf_output_close (zipdir);
}

static void 
create_info (PgtController *self,
	     GsfOutfile    *odir, 
	     gchar const   *album_date,
	     gchar const   *album_desc)
{
	GsfOutput *info = NULL;

	info = gsf_outfile_new_child (odir, "info.txt", FALSE);

	gsf_output_printf (info, "date|%s\n", album_date);
	gsf_output_printf (info, "description|%s\n", album_desc);
	gsf_output_close (info);
}

gboolean 
pgt_controller_convert (PgtController *self)
{
	const gchar *photo_dir = NULL;
	const gchar *album_name = NULL;
	const gchar *album_desc = NULL;
	const gchar *album_date = NULL;
	GDir *idir = NULL;
	gchar *basedir = NULL;
	GsfOutfile *odir = NULL;
	guint num_imgs;
	GError *err = NULL;
	gboolean options[num_options];

/* TODO
gchar const * pgt_model_get_album_name (PgtModel *self);
gchar const *  (PgtModel *self);
gchar const *  (PgtModel *self);
*/
	
	photo_dir = pgt_model_get_photo_dir (self->model);
	album_name = pgt_model_get_album_name (self->model);
	album_desc = pgt_model_get_album_desc (self->model);
	album_date = pgt_model_get_album_date (self->model);	
	options[option_lq] = pgt_model_get_option (self->model, option_lq);
	options[option_mq] = pgt_model_get_option (self->model, option_mq);
	options[option_hq] = pgt_model_get_option (self->model, option_hq);
	options[option_zip] = pgt_model_get_option (self->model, option_zip);
	options[option_comment] = pgt_model_get_option (self->model, option_comment);

	idir = g_dir_open (photo_dir, 0, &err);
	if (err) {
		g_error (err->message);
	}

	basedir = g_build_path (G_DIR_SEPARATOR_S, g_get_home_dir (), album_name, NULL);
	odir = gsf_outfile_stdio_new (basedir, &err);
	if (err) {
		g_error (err->message);
	}

	pgt_view_set_progress_text (self->view, _("converting ..."));
	gtk_main_iteration ();

	num_imgs = count_imgs (idir);
	create_gallery (self, photo_dir, idir, num_imgs, odir, options, GDK_INTERP_HYPER);
	create_info (self, odir, album_date, album_desc);
	gsf_output_close (GSF_OUTPUT (odir));

	pgt_view_set_progress_text (self->view, _("Conversion completed"));
	gtk_main_iteration ();

	pgt_controller_upload (self, basedir);
	
	g_free (basedir);

	return 0;
}

gboolean     
pgt_controller_read_config (PgtController *self)
{
	gchar *config_file = NULL;
	GsfInput *inp = NULL;
	GsfInputTextline *textinp = NULL;
	GError *err = NULL;
	guint8 *line = NULL;
	gchar url[1024], name[1024];

	config_file = g_build_path (G_DIR_SEPARATOR_S, g_get_home_dir (), ".plagiat", NULL);
	inp = gsf_input_stdio_new (config_file, &err);
	if (err) {
		/* TODO show gallery dialog?*/
		g_warning (err->message);
		return FALSE;
	}
	
	textinp = (GsfInputTextline *) gsf_input_textline_new (inp);
	g_object_unref (inp);
	inp = NULL;
	
	while (NULL != (line = gsf_input_textline_utf8_gets (textinp))) {

		sscanf ((gchar *) line, "%s %s", url, name);
		printf ("%s %s\n", url, name);
		pgt_model_add_gallery (self->model, name, url);
	}

	g_object_unref (textinp);
	textinp = NULL;

	return TRUE;
}

static void
walk (gchar const *basedir, GSList **entries)
{
	GError *err = NULL;
	GDir *dir = g_dir_open (basedir, 0, &err);
	const gchar *entry = NULL;
	gchar *path = NULL;

	while (NULL != (entry = g_dir_read_name (dir))) {

		path = g_build_path (G_DIR_SEPARATOR_S, basedir, entry, NULL);
		*entries = g_slist_prepend (*entries, path);
		if (g_file_test (path, G_FILE_TEST_IS_DIR)) {
			walk (path, entries);
		}
	}

	g_dir_close (dir);
}

gboolean
pgt_controller_upload (PgtController *self, gchar const *basedir)
{
	GSList *list = NULL;
	GSList *first = NULL;

	list = g_slist_prepend (list, g_strdup (basedir));
	walk (basedir, &list);

	if (!list) {
		return FALSE;
	}

	list = g_slist_reverse (list);
	first = list;

	do {
		printf ("%s\n", (gchar *) list->data);
		g_free (list->data);
	}
	while (NULL != (list = list->next));

	g_slist_free (first);

	return TRUE;
}
