;;; planner-experimental.el --- Experimental functions for Emacs planner mode
;;; Experimental functions for planner.el

;;; Commentary:
;;
;; Version: $Version$

(require 'planner)

;; Testing arch
;;;_* Notes

;;;_ + Linking through emacs-wiki links

;;; Code:
(defun planner-update-note ()
  "Copy the text from this note to the linked note, if any."
  (interactive)
  (save-restriction
    (save-excursion
      (save-window-excursion
        (when (planner-narrow-to-note)
          (let ((body (buffer-substring-no-properties (point-min) (point-max)))
                (info (planner-current-note-info)))
            ;; Get rid of the anchor.
            (when (string-match "^\\.#[0-9]+\\s-+" body)
              (setq body (replace-match "" nil t body)))
            ;; Get rid of the link
            (when (string-match "\\s-+(\\[\\[.+?\\]\\])" body)
              (setq body (replace-match "" nil t body)))
            ;; Insert the new body
            (when (planner-jump-to-linked-note)
              (save-restriction
                (save-excursion
                  (when (planner-narrow-to-note)
                    (goto-char (point-min))
                    (skip-chars-forward ".#0-9")
                    (delete-region (point) (point-max))
                    (insert " " body)
                    (goto-char (point-min))
                    (goto-char (line-end-position))
                    (insert " ([[" (planner-note-page info) "#"
                            (planner-note-anchor info) "]])")))))))))))

;;;_ + Searching

(defun planner-search-notes-next-match ()
  "Jump to the next matching entry.  Call after `planner-search-notes.'"
  (interactive)
  (if (buffer-live-p (get-buffer planner-search-notes-buffer))
      (progn
        (set-buffer planner-search-notes-buffer)
        (forward-line -1)
        (goto-char (line-beginning-position))
        (emacs-wiki-follow-name-at-point))
    (error "No current search.")))

(defun planner-search-notes-previous-match ()
  "Jump to the previous matching entry.  Call after `planner-search-notes.'"
  (interactive)
  (if (buffer-live-p (get-buffer planner-search-notes-buffer))
      (progn
        (set-buffer planner-search-notes-buffer)
        (forward-line 1)
        (goto-char (line-beginning-position))
        (emacs-wiki-follow-name-at-point))
    (error "No current search.")))

;;;_* Tasks

(defun planner-remove-duplicates ()
  "Remove duplicate tasks."
  (interactive)
  (goto-char (point-min))
  (let ((today (planner-today))
        (on-date (string-match planner-date-regexp (planner-page-name))))
    (while (re-search-forward "^#[A-C][0-9]*\\s-+\\(.+\\)$" nil t)
      (save-excursion
        (let* ((task-info (planner-current-task-info))
               (keep (planner-task-date task-info))
               date
               found
               (start (match-beginning 0)))
          (goto-char (line-beginning-position))
          (save-excursion
            (unless on-date
              (while (planner-find-task task-info (point))
                ;; Determine the most recent one
                (setq date (planner-task-date (planner-current-task-info)))
                (when (or (and (string< keep today)
                               (string< keep date))
                          (string< date keep))
                (setq keep date))
                (forward-line 1))))
          (while (planner-find-task task-info (point))
            (if (string= keep
                         (planner-task-date (planner-current-task-info)))
                (if found
                    (delete-region (line-beginning-position)
                                   (min (1+ (line-end-position))
                                        (point-max)))
                  (setq found t)
                  (forward-line 1))
              (planner-delete-task))))))))

;;;_* Indexing
;; I want to compress the list of day pages. Arranging them by month
;; may be a good strategy, although a calendar would be optimal.

(defun planner-index ()
  "Display an index of all known Wiki pages."
  (interactive)
  (let ((emacs-wiki-project planner-project))
    (message "Generating Wiki index...")
    (pop-to-buffer (planner-generate-index))
    (goto-char (point-min))
    (planner-mode)
    (message "Generating Wiki index...done")))

(defun planner-generate-index (&optional as-list exclude-private)
  "Generate an index of all Wiki pages.
List planner pages separately."
  (with-current-buffer (get-buffer-create "*Wiki Index*")
    (erase-buffer)
    (cd planner-directory)
    (emacs-wiki-change-project planner-project)
    (let ((files (emacs-wiki-index-files-list t))
          dates month last-month)
      (while files
        (unless (and exclude-private
                     (emacs-wiki-private-p
                      (emacs-wiki-index-file-page (car files))))
          (if (string-match
               planner-date-regexp
               (emacs-wiki-index-file-page (car files)))
              (setq dates
                    (append
                     (list (emacs-wiki-index-file-page (car files)))
                     dates))
            (insert
             (if as-list "- " "")
             (emacs-wiki-make-link
              (emacs-wiki-index-file-page (car files))
              (emacs-wiki-index-file-title (car files)))
             "\n")))
        (setq files (cdr files)))
      (when dates
        (insert "\n")
;;        (setq dates (nreverse dates))
        (while dates
          (setq month (substring (car dates) 0 7))
          (unless (string= month last-month)
            (setq last-month month)
            (insert "\n" month " |"))
          (insert " [[" (car dates) "][."
                  (substring (car dates) 8)
                  " ]]")
          (setq dates (cdr dates)))))
    (current-buffer)))

(defadvice emacs-wiki-generate-index (around planner-experimental activate)
  "Create a separate index for planner day pages."
  (setq ad-return-value (if (equal major-mode 'planner-mode)
                            (planner-generate-index (ad-get-arg 0)
                                                    (ad-get-arg 1))
                          ad-do-it)))

;;;_* Local emacs vars.
;;;Local variables:
;;;allout-layout: (-1 0 : )
;;;End:

(provide 'planner-experimental)

;;; planner-experimental.el ends here
