/*
 *   Copyright (C) 2007 Matt Broadstone <mbroadst@kde.org>
 *   Copyright (C) 2007 Matias Costa <m.costacano@gmail.com>
 *   Copyright (C) 2007 Montel Laurent <montel@kde.org>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License version 2 as
 *   published by the Free Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include <QDesktopWidget>
#include <QFileSystemWatcher>
#include <QApplication>
#include <QPainter>
#include <QGraphicsSceneMouseEvent>
#include <QGraphicsScene>
#include <QTimer>
#include <QAction>

#include <KDebug>
#include <KGlobalSettings>
#include <KAuthorized>
#include <KToggleAction>
#include <KFileItem>
#include <KDirLister>
#include <KIO/ListJob>
#include <KIO/Job>
#include <KIO/JobUiDelegate>
#include <KIO/UDSField>

#include "desktop.h"
#include "desktop.moc"
#include "launcher.h"
#include "desktoporganizer.h"
#include "plasmadesktopsettings.h"
#include "init.h"

StandardDesktop::StandardDesktop(QObject *parent, const QVariantList &args)
    : Plasma::Applet(parent, args),
      m_dirLister(0), m_solidEngine(0)
{
    setFlag(QGraphicsItem::ItemIsMovable, false);
    PlasmaDesktopSettings::instance("plasma-desktop");
    testLocalInstallation();
    // Desktop files
    m_dirLister = new KDirLister;
    connect(m_dirLister, SIGNAL(clear()), this, SLOT(clearDesktop()));
    connect(m_dirLister, SIGNAL(completed()), this, SLOT(listingCompleted()));
    connect(m_dirLister, SIGNAL(newItems(const KFileItemList&)),
            this, SLOT(newItemsFound(const KFileItemList&)));
    connect(m_dirLister, SIGNAL(deleteItem(KFileItem*)),
            this, SLOT(itemDeleted(KFileItem*)));

    KUrl desktopUrl(KGlobalSettings::desktopPath());
    KAuthorized::allowUrlAction("list", KUrl(), desktopUrl);
    m_dirLister->setShowingDotFiles(PlasmaDesktopSettings::showHidden);
    m_dirLister->openUrl(desktopUrl);


    // Solid
    m_enableMedia = PlasmaDesktopSettings::mediaEnabled();
    
    // Organization
    m_organizer = new DesktopOrganizer(this);
    configureMedia();

    //create menu
    createMenu();
    
    // Grid
    m_gridAlign = true;
    m_bLockIcon = !PlasmaDesktopSettings::lockIcons;
    setGridSize(QSizeF(100, 100));
    readIconPosition();

}

StandardDesktop::~StandardDesktop()
{
    saveIconPosition();
    delete m_solidEngine;
    delete m_dirLister;
}

void StandardDesktop::createMenu()
{
    QAction* alignHorizontal = new QAction(i18n("Line Up Horizontal"), this);
    actions.append(alignHorizontal);
    connect(alignHorizontal, SIGNAL(triggered(bool)), this , SLOT(slotAlignHorizontal()));
    QAction* alignVertical = new QAction(i18n("Line Up Vertical"), this);
    connect(alignVertical, SIGNAL(triggered(bool)), this , SLOT(slotAlignVertical()));
    actions.append(alignVertical);

    QAction* separator = new QAction(this);
    separator->setSeparator(true);
    actions.append(separator);
    m_lockIcon = new KToggleAction(i18n("Lock"),this);
    connect(m_lockIcon, SIGNAL(triggered(bool)), this , SLOT(slotLock(bool)));
    actions.append(m_lockIcon);
    m_alignToGrid = new KToggleAction(i18n("Align to grid"),this);
    connect(m_alignToGrid, SIGNAL(triggered(bool)), this , SLOT(slotAlignToGrid(bool)));
    actions.append(m_alignToGrid);

    separator = new QAction(this);
    separator->setSeparator(true);
    actions.append(separator);
    
    QAction* sortByNameSensitive = new QAction(i18n("Sort by Name (Case Sensitive)"), this);
    actions.append(sortByNameSensitive);
    connect(sortByNameSensitive, SIGNAL(triggered(bool)), this , SLOT(slotSortByNameCaseSensitive()));
    QAction* sortByNameInsensitive = new QAction(i18n("Sort by Name (Case Insensitive)"), this);
    actions.append(sortByNameInsensitive);
    connect(sortByNameInsensitive, SIGNAL(triggered(bool)), this , SLOT(slotSortByNameCaseInsensitive()));
    QAction* sortBySize = new QAction(i18n("Sort by Size"), this);
    actions.append(sortBySize);
    connect(sortBySize, SIGNAL(triggered(bool)), this , SLOT(slotSortBySize()));
    QAction* sortByType = new QAction(i18n("Sort by Type"), this);
    actions.append(sortByType);
    connect(sortByType, SIGNAL(triggered(bool)), this , SLOT(slotSortByType()));


}

QList<QAction*> StandardDesktop::contextActions()
{
  //Update action.
  m_alignToGrid->setChecked(m_gridAlign);
  m_lockIcon->setChecked(m_bLockIcon);
  return actions;
}

void StandardDesktop::slotSortByNameCaseSensitive()
{
}

void StandardDesktop::slotSortByNameCaseInsensitive()
{
}

void StandardDesktop::slotSortBySize()
{
}

void StandardDesktop::slotSortByType()
{
}

void StandardDesktop::slotSortByDate()
{
}


void StandardDesktop::slotAlignToGrid(bool b)
{
  setGridAligned(b);
}

void StandardDesktop::slotLock(bool lock)
{
   m_bLockIcon = !lock;
   foreach (Launcher *item, m_desktopItems)
        item->lockIcon(m_bLockIcon);
}

void StandardDesktop::slotAlignHorizontal()
{
  m_organizer->changeAlignment(DesktopOrganizer::Horizontal);
  m_organizer->organize(m_desktopItems);
}

void StandardDesktop::slotAlignVertical()
{
  m_organizer->changeAlignment(DesktopOrganizer::Vertical);
  m_organizer->organize(m_desktopItems);
}

void StandardDesktop::changeAlignment(bool horizontal)
{
  if(horizontal)
    m_organizer->changeAlignment(DesktopOrganizer::Horizontal);
  else
    m_organizer->changeAlignment(DesktopOrganizer::Vertical);
}

void StandardDesktop::configureMedia()
{
    if(m_enableMedia)
    {
       if(!m_solidEngine)
       {
           m_solidEngine = dataEngine("solidnotifierengine");
           connect(m_solidEngine, SIGNAL(newSource(const QString&)),
                     this, SLOT(sourceAdded(const QString&)));
           //TODO update it.
       }
    }
    else
    {
       if(m_solidEngine)
       {
          //TODO delete all icon in data
          delete m_solidEngine;
          m_solidEngine=0L;
          m_solidDevices.clear();
       }
    } 
}

bool StandardDesktop::isGridAligned() const
{
    return m_gridAlign;
}

void StandardDesktop::setGridAligned(bool align)
{
    m_gridAlign = align;
}

void StandardDesktop::alignToGrid(Launcher *item)
{
    QPointF currentPos = mapToParent(item->pos());
    QPointF currentGridPos = mapToGrid(currentPos);
    alignToGrid(item, currentGridPos);
}

void StandardDesktop::alignToGrid(Launcher *item, const QPointF &pos)
{
    qreal width = item->geometry().width();
    qreal height = item->geometry().height();

    QPointF scenePos = mapFromGrid(pos.toPoint());
    scenePos.rx() -= width/2;
    scenePos.ry() -= height/2;
    item->setPos(scenePos);
}

QSizeF StandardDesktop::gridSize() const
{
    return m_gridSize;
}

QSize StandardDesktop::gridDimensions() const
{
    QSizeF thisSize = contentSize();
    return QSize( int(thisSize.width() / m_gridSize.width()),
                  int(thisSize.height() / m_gridSize.height()) );
}

void StandardDesktop::setGridSize(const QSizeF& gridSize)
{
    QSizeF desktopSize = contentSize();
    qreal bestMachWidth = desktopSize.width()
                        / qRound(desktopSize.width()/gridSize.width());
    qreal bestMachHeight = desktopSize.height()
                        / qRound(desktopSize.height()/gridSize.height());
    m_gridSize = QSizeF(bestMachWidth, bestMachHeight);
    if (m_gridAlign) {
        foreach(Launcher *item, m_desktopItems) {
            alignToGrid(item);
        }
    }
}

QSizeF StandardDesktop::contentSizeHint() const
{
    // This should probably use KWin, this is easy right now..
    // TODO fix for multihead
     return QSizeF(QApplication::desktop()->screenGeometry().size());
}

void StandardDesktop::paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
                            const QRect &rect)
{
    Q_UNUSED(option);
    painter->setPen(Qt::red);
    painter->drawRect(rect);
/*    QSizeF desktopSize = contentSize();
    for (qreal i=m_gridSize.width(); i<desktopSize.width(); i+=m_gridSize.width()) {
        painter->drawLine(QPointF(i, 0), QPointF(i, desktopSize.height()));
    }
    for (qreal i=m_gridSize.height(); i<desktopSize.height(); i+=m_gridSize.height()) {
        painter->drawLine(QPointF(0, i), QPointF(desktopSize.width(), i));
    }*/
}

void StandardDesktop::clearDesktop()
{
    foreach (Launcher *item, m_desktopItems)
        item->deleteLater();

    m_desktopItems.clear();
    m_fileItems.clear();
    m_solidDevices.clear();

    // ^^^^ This is incomplete
}

void StandardDesktop::listingCompleted()
{
    // we may eventually want to organize on this call
}

void StandardDesktop::newItemsFound(const KFileItemList &items)
{
    QList<Launcher*> newLaunchers;
    foreach (KFileItem *item, items)
    {
        if ( (item->isDir() && (item->name() == "." || item->name() == "..")) ||
                item->isHidden())
            continue;

        if (m_fileItems.contains(item))    // Find a better way to hash this
            continue;

        FileLauncher *tmp = new FileLauncher(item, this);
        tmp->installSceneEventFilter(this);

        // kDebug() << "New launcher: " << tmp->size() << tmp->boundingRect() << endl;
        newLaunchers << tmp;
        m_desktopItems << tmp;
        m_fileItems[item] = tmp;

        tmp->setPos(100, 100);
    }
    m_organizer->organize(newLaunchers);
}

void StandardDesktop::itemDeleted(KFileItem *file)
{
    Launcher *item = m_fileItems[file];
    item->deleteLater();
}

void StandardDesktop::updated(const QString &source, Plasma::DataEngine::Data data)
{
    Q_UNUSED(source);
    QString deviceUdi = data["udi"].toString();
    bool newDevice = data["added"].toBool();

    kDebug() << "new device: added[" << newDevice << "] udi[" << deviceUdi << "]";

    if (newDevice)
    {
        if (!m_solidDevices[deviceUdi])
        {
            DeviceLauncher *launcher = new DeviceLauncher(deviceUdi, this);
            launcher->installSceneEventFilter(this);

            launcher->setPos(100, 100);
        }
        else
        {
            kDebug() << "added twice?";
        }
    }
    else
    {
        DeviceLauncher *launcher = m_solidDevices[deviceUdi];
        if (launcher)
            launcher->deleteLater();
    }
}

void StandardDesktop::sourceAdded(const QString &source)
{
    kDebug() << "sourceAdded";
    m_solidEngine->connectSource(source, this);
}

bool StandardDesktop::sceneEventFilter(QGraphicsItem *target, QEvent *event)
{
    switch (event->type())
    {
        case QEvent::GraphicsSceneMouseRelease:
            {
                if (m_gridAlign)
                {
                    foreach (QGraphicsItem *item, scene()->selectedItems())
                    {
                        Launcher *launcher = qgraphicsitem_cast<Launcher*>(item);
                        if (launcher)
                            alignToGrid(launcher);
                    }
                }
            }
            break;

        default:
            break;
    }

    return QGraphicsItem::sceneEventFilter(target, event);
}

void StandardDesktop::saveIconPosition()
{
}

void StandardDesktop::readIconPosition()
{
}

K_EXPORT_PLASMA_APPLET(desktop, StandardDesktop)
