/***********************************************************************************
* Fancy Tasks: Plasmoid for fancy representing your tasks and launchers.
* Copyright (C) 2009 Michal Dutkiewicz aka Emdek <emdeck@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "FancyTasksLight.h"
#include "FancyTasksIcon.h"
#include "FancyTasksTask.h"

#include <KMenu>
#include <KIconLoader>
#include <KIconEffect>

#include <Plasma/ToolTipManager>

FancyTasksLight::FancyTasksLight(WId window, Plasma::Svg *svg, FancyTasksIcon *icon) : QGraphicsWidget(icon),
    m_taskIcon(icon),
    m_task(NULL),
    m_svg(svg),
    m_window(window),
    m_dragTimer(0),
    m_highlightTimer(0)
{
    setAcceptsHoverEvents(true);

    setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));

    setFlag(QGraphicsItem::ItemIsFocusable);

    Plasma::ToolTipManager::self()->registerWidget(this);

    TaskManager::TaskPtr task = TaskManager::TaskManager::self()->findTask(m_window);

    if (!task)
    {
        deleteLater();
    }

    m_task = new FancyTasksTask(new TaskManager::TaskItem(this, task), new TaskManager::GroupManager(this), icon);

    connect(m_taskIcon, SIGNAL(sizeChanged(qreal)), this, SLOT(setSize(qreal)));
    connect(m_taskIcon, SIGNAL(colorChanged(QColor)), this, SLOT(setColor(QColor)));
}

void FancyTasksLight::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_UNUSED(option)
    Q_UNUSED(widget)

    QPixmap pixmap(boundingRect().size().toSize());
    pixmap.fill(Qt::transparent);

    QPainter pixmapPainter(&pixmap);
    pixmapPainter.setRenderHints(QPainter::SmoothPixmapTransform | QPainter::Antialiasing | QPainter::TextAntialiasing);

    m_svg->paint(&pixmapPainter, 0, 0, pixmap.width(), pixmap.height(), "task");

    pixmapPainter.end();

    if (m_color.isValid())
    {
        pixmap = KIconLoader::global()->iconEffect()->apply(pixmap, KIconEffect::Colorize, 1, m_color, true);
    }

    painter->drawPixmap(QPointF(0, 0), pixmap, boundingRect());
}

void FancyTasksLight::hoverEnterEvent(QGraphicsSceneHoverEvent *event)
{
    Q_UNUSED(event)

    Plasma::ToolTipContent data;
    data.setMainText(m_task->title());
    data.setSubText(m_task->description());
    data.setImage(m_task->icon());
    data.setWindowToPreview(m_window);

    Plasma::ToolTipManager::self()->setContent(this, data);

    m_highlightTimer = startTimer(500);

    setOpacity(0.7);

    update();
}

void FancyTasksLight::hoverLeaveEvent(QGraphicsSceneHoverEvent *event)
{
    Q_UNUSED(event)

    killTimer(m_highlightTimer);

    setOpacity(1);

    update();

    emit stopWindowsHighlight(m_task->windows());
}

void FancyTasksLight::dragEnterEvent(QGraphicsSceneDragDropEvent *event)
{
    Q_UNUSED(event)

    killTimer(m_dragTimer);

    m_dragTimer = startTimer(300);

    update();
}

void FancyTasksLight::dragLeaveEvent(QGraphicsSceneDragDropEvent *event)
{
    Q_UNUSED(event)

    killTimer(m_dragTimer);

    update();
}

void FancyTasksLight::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    event->accept();
}

void FancyTasksLight::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    if (event->button() == Qt::LeftButton)
    {
       m_task->activate();

        event->accept();
    }
    else if (event->button() == Qt::MidButton)
    {
        m_task->close();

        event->accept();
    }
}

void FancyTasksLight::contextMenuEvent(QGraphicsSceneContextMenuEvent *event)
{
    KMenu *menu = m_task->contextMenu();
    menu->addTitle(KIcon(m_task->icon()), m_task->title().left(20), menu->actions().at(0));
    menu->exec(QCursor::pos());

    delete menu;

    event->accept();
}

void FancyTasksLight::timerEvent(QTimerEvent *event)
{
    if (event->timerId() == m_dragTimer && isUnderMouse())
    {
        m_task->activateWindow(true);
    }
    else if (event->timerId() == m_highlightTimer)
    {
        emit startWindowsHighlight(m_task->windows());
    }

    killTimer(event->timerId());
}

void FancyTasksLight::setSize(qreal size)
{
    size *= 0.1;

    setPreferredSize(size, size);
}

void FancyTasksLight::setColor(QColor color)
{
    m_color = color;

    update();
}
