/***************************************************************************
 *   Copyright (C) 2008 by Damien Lévin <dml_aon@hotmail.com>      	   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .        *
 ***************************************************************************/


#include "lastmoid.h"


#include <plasma/svg.h>
#include <plasma/theme.h>
#include <QDebug> 
#include <QDomDocument>
#include <QDomElement>
#include <QRectF>
#include <QPainter>
#include <KConfigDialog>


Lastmoid::Lastmoid(QObject *parent, const QVariantList &args) : Plasma::Applet(parent, args) 

{

    m_svg.setImagePath("widgets/lastmoid");
    timer = new QTimer(this);

    connect(&http, SIGNAL(requestFinished(int, bool)),this, SLOT(finished(int, bool)));
    connect(timer, SIGNAL(timeout()), this, SLOT(refresh()));

    
    setBackgroundHints(DefaultBackground);
    resize(200, 220);
    refresh();

}
 
 
Lastmoid::~Lastmoid()
{
   timer->stop();
   delete timer;
}


void Lastmoid::refresh()
{
   qDebug() << icon();
   fetch();
   timer->stop();
   timer->setInterval(updateFrequency * 60 * 1000);
   timer->start();
}


void Lastmoid::createConfigurationInterface(KConfigDialog *parent)
{

	QWidget *widgetConfig = new QWidget;
	configGroup = config();
	uiConfig.setupUi(widgetConfig);
	uiConfig.user->setText(configGroup.readEntry("user"));
	uiConfig.nbDatas->setValue(QString(configGroup.readEntry("nbDatas")).toInt());
	uiConfig.dataType->setCurrentIndex(uiConfig.dataType->findText(configGroup.readEntry("dataType")));
	uiConfig.dataPeriod->setCurrentIndex(uiConfig.dataPeriod->findText(configGroup.readEntry("dataPeriod")));

	connect(parent, SIGNAL(okClicked()), this, SLOT(configAccepted()));
	parent->setButtons(KDialog::Ok | KDialog::Cancel);
	parent->addPage(widgetConfig, i18n("Configuration"), icon());

}



void Lastmoid::configAccepted(){
    
    lastUser = uiConfig.user->text();
    nbDatas = uiConfig.nbDatas->value();
    dataType = uiConfig.dataType->currentText();
    dataPeriod = uiConfig.dataPeriod->currentText();

    configGroup = config();
    configGroup.writeEntry("user", lastUser);
    configGroup.writeEntry("nbDatas",QString::number(nbDatas));
    configGroup.writeEntry("dataType",dataType);
    configGroup.writeEntry("dataPeriod",dataPeriod);
    

    updateFrequency = 30;
    refresh();
}


void Lastmoid::fetch()
 {

    configGroup = config();
    lastUser = configGroup.readEntry("user");
    nbDatas = QString(configGroup.readEntry("nbDatas")).toInt();
    dataType = configGroup.readEntry("dataType");
    dataPeriod = configGroup.readEntry("dataPeriod");

    if (QString::compare(dataPeriod,QString("weekly"))==0){
        url.setUrl("http://ws.audioscrobbler.com/2.0/user/"+lastUser+"/weekly"+dataType+"chart.xml");
    }else{
        url.setUrl("http://ws.audioscrobbler.com/2.0/user/"+lastUser+"/top"+dataType+"s.xml?period="+dataPeriod);
    }
    qDebug() << url.toString ();
    datas.clear();

    http.setHost(url.host());
    connectionId = http.get(url.toString ());
 }


 void Lastmoid::finished(int id, bool error)
 {
     if (error) {
         qWarning("Received error during HTTP fetch.");
     }else{
	parseXml();
        update();
    }
 }

void Lastmoid::parseXml()
 {
 
    QDomDocument doc("?xml version=\"1.0\" encoding=\"UTF-8\" ?");
    QDomElement name,playcount,root,element;

    doc.setContent (http.readAll());

    if (dataPeriod =="weekly"){
      root = doc.firstChildElement("weekly"+dataType+"chart");
    }else{
      root = doc.firstChildElement("top"+dataType+"s");
    }
    element = root.firstChildElement(dataType);

  
    for (; !element.isNull(); element = element.nextSiblingElement(dataType)) {

	QStringList item;
	
	name = element.firstChildElement("name");
	playcount = element.firstChildElement("playcount");

	item.append(element.attribute("rank"));
	item.append(name.text());
	item.append(playcount.text());
	datas.append(item);

   }

 }

void Lastmoid::paintInterface(QPainter *p, const QStyleOptionGraphicsItem *option, const QRect &contentsRect)
{

    int i=1,widgetWidth,widgetHeight,vAlign,hAlign,maxListening=0;
    QRectF rect;

    widgetWidth = (int)contentsRect.width();
    widgetHeight = (int)contentsRect.height()-60;
    
    
    p->setRenderHint(QPainter::SmoothPixmapTransform);
    p->setRenderHint(QPainter::Antialiasing);
 

     hAlign = widgetWidth/10;
     m_svg.resize(94, 48);
     m_svg.paint(p,hAlign, (int)contentsRect.top());    

    //Get maxListening 
    if(datas.size()>=1){	
	maxListening=datas.at(0).at(2).toInt();
    }



    for (i=0;i < nbDatas;i++){     

      if((datas.size() > i) ){

	vAlign = (i+1)*widgetHeight/nbDatas+60;
	
	rect = QRectF(hAlign,vAlign,widgetWidth-hAlign,-widgetHeight/nbDatas*0.75);
	p->fillRect(rect,QBrush(QColor(234, 234, 234, 127)));


	rect = QRectF(hAlign,vAlign,(widgetWidth-hAlign)*datas.at(i).at(2).toInt()/maxListening,-widgetHeight/nbDatas*0.75);
	p->setPen(Qt::red);
	p->fillRect(rect,QBrush(QColor(215, 0, 25, 200)));
	

	p->setPen(Qt::black);
	p->drawText(rect,Qt::AlignLeft | Qt::AlignVCenter| Qt::TextDontClip ,"  "+datas.at(i).at(0)+" - "+datas.at(i).at(1)+" ("+datas.at(i).at(2)+")");
	p->restore();
	
      }
    }
}
 
#include "lastmoid.moc"