/******************************************************************************
*                                PlayWolf                                     *
*******************************************************************************
*                                                                             *
*                   Copyright (C) 2008-2009 Giulio Camuffo		      *
*                                                                             *
*   This program is free software; you can redistribute it and/or modify      *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   This program is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License along   *
*   with this program; if not, write to the Free Software Foundation, Inc.,   *
*   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA                *
*                                                                             *
*                                                                             *
*                                                                             *
*   For further information contact me at giuliocamuffo@gmail.com	      *
******************************************************************************/


#include "wolflabel.h"

#include <QtGui/QStyleFactory>
#include <QtCore/QTimeLine>
#include <QtGui/QGraphicsItemAnimation>
#include <QtGui/QGraphicsSceneResizeEvent>
#include <QtGui/QLabel>

#include <KIO/Job>
#include <KIO/NetAccess>
#include <KTemporaryFile>

#include <Plasma/Label>

WolfLabel::WolfLabel(QGraphicsWidget *parent)
    : QGraphicsWidget(parent)
{
    setFlag(QGraphicsItem::ItemClipsChildrenToShape, true);

    label = new Plasma::Label(this);
    label->nativeWidget()->setAlignment(Qt::AlignCenter);
    label->nativeWidget()->setWordWrap(false);

    //workaround until is solved the bug at
    //http://trolltech.com/developer/task-tracker/index_html?method=entry&id=220488
    QStyleFactory styleFactory = QStyleFactory();
    QStyle *style = styleFactory.create("Oxygen");
    label->setStyle( style );

    myFont = label->nativeWidget()->font();

    timer = new QTimeLine(20000);
    timer->setFrameRange(0, 100);
    timer->setUpdateInterval(200); ///Find a good compromise
    timer->setLoopCount(0);

    animation = new QGraphicsItemAnimation;
    animation->setItem(label);
    animation->setTimeLine(timer);

    resizeFont = true;

}

WolfLabel::~WolfLabel() {
    setFlag(QGraphicsItem::ItemClipsChildrenToShape, false);
    timer->stop();
    delete timer;
    delete animation;
}

void WolfLabel::setText(const QString& text) {
    label->setText(text);
    updateLabel();
}

void WolfLabel::setImage(const QString& path) {
	KUrl url(path);

	QString nocover = "file:///usr/share/apps/amarok/images/nocover.png";
	KTemporaryFile tmpFile;
	if (tmpFile.open()) {
		KIO::Job* copyJob = KIO::file_copy( url, KUrl( tmpFile.fileName() ), -1, KIO::Overwrite | KIO::HideProgressInfo );
			if (KIO::NetAccess::synchronousRun(copyJob, 0))
				image.load( tmpFile.fileName() );
			else
				image.load( nocover );
	}
	else
		image.load( nocover );


	label->nativeWidget()->setPixmap(image.scaled(size().width(), size().height(), Qt::KeepAspectRatio, Qt::SmoothTransformation));
}

void WolfLabel::clear(void) {
    label->nativeWidget()->clear();
}

void WolfLabel::resizeEvent(QGraphicsSceneResizeEvent *event) {
    QGraphicsWidget::resizeEvent(event);
    label->resize(event->newSize());

    if (!label->nativeWidget()->pixmap()) {
	updateLabel();
    }
    else
	label->nativeWidget()->setPixmap(image.scaled(event->newSize().width(), event->newSize().height(), Qt::KeepAspectRatio, Qt::SmoothTransformation));
}

void WolfLabel::adjustFont() {
    QFont myFont = label->nativeWidget()->font();
    QFontMetrics metric(myFont);
    QString text = label->text();

    int width = size().width();

    qreal labelWidth = metric.width(text);

    while (labelWidth > width) {
	if (myFont.pixelSize() < 2)
	    break;
	myFont.setPixelSize(myFont.pixelSize()-1);
	QFontMetrics metric(myFont);
	labelWidth = metric.width(text);
    }

    QFontInfo f = QFontInfo(myFont);

    if (f.pointSize() >= minFont)
	label->nativeWidget()->setFont(myFont);
    else
	scrollLabel();
}

void WolfLabel::scrollLabel() {
    QSize sizeHint = label->widget()->sizeHint();
    if (sizeHint.width() > (contentsRect().width())) {

	label->resize(sizeHint);

	qreal y = label->y();
	qreal gap = (sizeHint.width() - contentsRect().width())+15;

	for (int i = 0; i < 100; ++i)
	    animation->setPosAt(i / 200.0, QPointF(-gap+(gap*(i/100.0))+5, y));
	for (int i = 0; i < 100; ++i)
	    animation->setPosAt((100+i) / 200.0, QPointF(-gap*(i/100.0)+5, y));

	timer->start();
    }
}

bool WolfLabel::isResizeFont() {
    return resizeFont;
}

void WolfLabel::setResizeFont(bool resize, int min) {
    resizeFont = resize;
    minFont = min;
    updateLabel();
}

int WolfLabel::minimumFont() {
    return minFont;
}

void WolfLabel::setFont(QFont customFont) {
    myFont = customFont;
}

QFont WolfLabel::font() {
    return myFont;
}

void WolfLabel::updateLabel(QString mode) {
    if (label->text() != QString()) {
	timer->stop();

	int height = size().height();
	QFont font = myFont;
	QFontInfo f = QFontInfo(font);
	int pixelSize = f.pixelSize();
	if (pixelSize > height) {
	    if (height <= 0)
		height = 1;
	    font.setPixelSize(height);
	}
	else {
	    if (pixelSize <= 0)
		pixelSize = 1;
	    font.setPixelSize(pixelSize);
	}
	label->nativeWidget()->setFont(font);

	label->resize(size());
	qreal qHeight = size().height();
	qreal y = (qHeight - label->widget()->sizeHint().height())/2.0;
	label->setPos(0,y);

	if (resizeFont or (mode == "resizeFont"))
	    adjustFont();
	else
	    scrollLabel();
    }
}

QSizeF WolfLabel::labelSize() {
    updateLabel("resizeFont");
    return label->widget()->sizeHint();
}

bool WolfLabel::contains(const QPointF& point) const {
    QRectF rect = geometry();

    return ((point.x() >= rect.x()) and (point.x() <= (rect.x() + rect.width())) and
	    (point.y() >= rect.y()) and (point.y() <= (rect.y() + rect.height())));
}

#include "wolflabel.moc"
