/***********************************************************************************
* STasks
* Copyright (C) 2009 Marcin Baszczewski <marcin.baszczewski@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "stasks_tooltip.h"
#include "stasks_windowpreview.h"

#include <Plasma/Theme>
#include <Plasma/IconWidget>
#include <KIcon>
#include <KIconLoader>
#include <QTimer>
#include <QPushButton>
#include <QAbstractButton>
#include <QGraphicsLinearLayout>
#include <QToolButton>

#ifdef Q_WS_X11
#include <QX11Info>

#include <X11/Xlib.h>
#include <fixx11h.h>
#endif

class QClickableLabel : public QLabel
{
    public:
	void setTask(TaskManager::TaskItem *task)
	{
	    m_task = task;
	}
    private:
	TaskManager::TaskItem *m_task;
    protected:
        void mousePressEvent(QMouseEvent *event)
	{
	    if (m_task->task())
	    m_task->task()->activate();
	    event->ignore();
	}
};

class CloseIcon : public QLabel
{
    public:
	void setTask(TaskManager::TaskItem *task)
	{
	    m_task = task;
	}
    private:
	TaskManager::TaskItem *m_task;
    protected:
        void mousePressEvent(QMouseEvent *event)
	{
	    m_task->close();
	    event->ignore();
	}
};

STasksToolTip::STasksToolTip(int size, QWidget *parent) : QWidget(parent)
{
    setWindowFlags(Qt::ToolTip);

    m_size = size;

    m_canHide = true;

    m_alreadyShown = false;

    m_previewsAvailable = false;

#ifdef Q_WS_X11
    if (KWindowSystem::compositingActive())
    {
        int number;
        Display *display = QX11Info::display();
        Atom atom = XInternAtom(display, "_KDE_WINDOW_PREVIEW", false);
        Atom *list = XListProperties(display, DefaultRootWindow(display), &number);

        if (list != NULL)
        {
            m_previewsAvailable = (qFind(list, (list + number), atom) != (list + number));

            XFree(list);
        }
    }
#endif

    m_tasksLayout = new QGridLayout;

    m_mainLayout = new QGridLayout(this);
    m_mainLayout->setSpacing(3);
    m_mainLayout->addLayout(m_tasksLayout, 0, 0);

    setLayout(m_mainLayout);

    m_background = new Plasma::FrameSvg(this);
    m_background->setImagePath("widgets/tooltip");
    m_background->setEnabledBorders(Plasma::FrameSvg::AllBorders);

    updateTheme();

    connect(m_background, SIGNAL(repaintNeeded()), this, SLOT(updateTheme()));
}

STasksToolTip::~STasksToolTip()
{
    delete m_tasksLayout;
    delete m_mainLayout;
}

void STasksToolTip::paintEvent(QPaintEvent *event)
{
    Q_UNUSED(event)

    QPainter painter(this);
    painter.setRenderHint(QPainter::Antialiasing);
    painter.setClipRect(event->rect());
    painter.setCompositionMode(QPainter::CompositionMode_Source);
    painter.fillRect(rect(), Qt::transparent);

    m_background->paintFrame(&painter);
}

void STasksToolTip::resizeEvent(QResizeEvent *event)
{
    QWidget::resizeEvent(event);

    m_background->resizeFrame(size());

    setMask(m_background->mask());
}

void STasksToolTip::enterEvent(QEvent *event)
{
    Q_UNUSED(event)

    m_canHide = false;
}

void STasksToolTip::leaveEvent(QEvent *event)
{
    Q_UNUSED(event)

    m_canHide = true;

    QTimer::singleShot(500, this, SLOT(hide()));
}

void STasksToolTip::mousePressEvent(QMouseEvent *event)
{
    m_canHide = true;

    hide();

    event->accept();
}

void STasksToolTip::setPreview(QPixmap pixmap, WId window)
{
    if (m_previews.count())
    {
        m_previews.at(0)->deleteLater();
        m_previews.removeAt(0);
    }

    if (m_previewsAvailable && window)
    {
        QSize windowSize = KWindowSystem::windowInfo(window, NET::WMGeometry | NET::WMFrameExtents).frameGeometry().size();

        if (windowSize.isValid())
        {
            if (windowSize.width() > m_size || windowSize.height() > m_size)
            {
                windowSize.scale(m_size, m_size, Qt::KeepAspectRatio);
            }

            pixmap = QPixmap(windowSize);
            pixmap.fill(Qt::transparent);
        }
    }

    if (!pixmap.isNull())
    {
        STasksWindowPreview *preview = new STasksWindowPreview(pixmap, this);

        if (window)
        {
            preview->setWindow(window);
        }

        m_previews.append(preview);

        m_tasksLayout->addWidget(preview, 0, 0, Qt::AlignCenter);
    }

    adjustSize();
}

void::STasksToolTip::setTasks(TaskManager::ItemList tasks)
{
    for (int i = 0; i < m_tasksLayout->count(); ++i)
    {
        m_tasksLayout->itemAt(i)->widget()->deleteLater();
    }

    m_mainLayout->removeItem(m_tasksLayout);

    delete m_tasksLayout;

    m_tasksLayout = new QGridLayout;

    m_mainLayout->addLayout(m_tasksLayout, 0, 0);

    for (int i = 0; i < tasks.count(); ++i)
    {
        TaskManager::TaskItem *task = static_cast<TaskManager::TaskItem*>(tasks.at(i));

	QLabel *iconLabel = new QLabel;
	iconLabel->setMaximumWidth(20);
        iconLabel->setPixmap(tasks.at(i)->icon().pixmap(16, 16));
      
        m_tasksLayout->addWidget(iconLabel, 0, (i*3),1,1);

	QClickableLabel *label = new QClickableLabel;
	label->setText(tasks.at(i)->name());
	label->setMaximumWidth(160);
	label->setTask(task);
        m_tasksLayout->addWidget(label, 0, (i*3)+1,1,1);

	CloseIcon *iconClose = new CloseIcon;
	iconClose->setMaximumWidth(20);
	iconClose->setTask(task);
	iconClose->setPixmap(KIcon("dialog-close").pixmap(16, 16));
        m_tasksLayout->addWidget(iconClose, 0, (i*3)+2,1,1);

        if (m_previewsAvailable)
        {
            QPixmap pixmap;

            if (task->task()->window())
            {
                QSize windowSize = KWindowSystem::windowInfo(task->task()->window(), NET::WMGeometry | NET::WMFrameExtents).frameGeometry().size();

                if (windowSize.isValid())
                {
                    if (windowSize.width() > m_size || windowSize.height() > m_size)
                    {
                        windowSize.scale(m_size, m_size, Qt::KeepAspectRatio);
                    }

                    pixmap = QPixmap(windowSize);
                    pixmap.fill(Qt::transparent);
                }
            }

            if (!pixmap.isNull())
            {
                STasksWindowPreview *preview = new STasksWindowPreview(pixmap, this);
                preview->setWindow(task->task()->window());

                m_previews.append(preview);

                m_tasksLayout->addWidget(preview, 1, i*3,1,3);
            }
        }

        
    }

    adjustSize();
}

void STasksToolTip::show()
{
#ifdef Q_WS_X11
    if (m_alreadyShown)
    {
        QWidget::show();

        return;
    }

    m_alreadyShown = true;

    for (int i = 0; i < m_previews.count(); ++i)
    {
        m_previews.at(i)->show();
    }

    m_tasksLayout->activate();

    Display *display = QX11Info::display();
    long unsigned int atom = XInternAtom(display, "_KDE_WINDOW_PREVIEW", false);
    long data[(m_previews.count() * 6) + 1];
    data[0] = m_previews.count();

    for (int i = 0; i < m_previews.count(); ++i)
    {
        int index = (i * 6);

        qreal left, top, right, bottom;

        m_previews.at(i)->m_background->getMargins(left, top, right, bottom);

        QRect thumbnailRect = m_previews.at(i)->geometry().adjusted(left, top, -right, -bottom);

        data[index + 1] = 5;
        data[index + 2] = m_previews.at(i)->m_window;
        data[index + 3] = thumbnailRect.left();
        data[index + 4] = thumbnailRect.top();
        data[index + 5] = thumbnailRect.width();
        data[index + 6] = thumbnailRect.height();
    }

    XChangeProperty(display, winId(), atom, atom, 32, PropModeReplace, reinterpret_cast<unsigned char*>(data), (sizeof(data) / sizeof(data[0])));
#endif

    QWidget::show();
}

void STasksToolTip::hide(bool force)
{
    if (m_canHide || force)
    {
        m_canHide = true;

        QWidget::hide();
    }
}

void STasksToolTip::updateTheme()
{
    m_background->clearCache();

    m_mainLayout->setContentsMargins(m_background->marginSize(Plasma::LeftMargin), m_background->marginSize(Plasma::TopMargin), m_background->marginSize(Plasma::RightMargin), m_background->marginSize(Plasma::BottomMargin));

    QPalette plasmaPalette = QPalette();
    plasmaPalette.setColor(QPalette::Window, Plasma::Theme::defaultTheme()->color(Plasma::Theme::BackgroundColor));
    plasmaPalette.setColor(QPalette::WindowText, Plasma::Theme::defaultTheme()->color(Plasma::Theme::TextColor));

    setAutoFillBackground(true);

    setPalette(plasmaPalette);

    update();
}
