/***********************************************************************************
* System Monitor: Plasmoid and data engines to monitor CPU/Memory/Swap Usage.
* Copyright (C) 2008  Matthew Dawson
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "system_monitor.h"

#include <QPainter>
#include <KConfigDialog>

systemMonitor::systemMonitor(QObject *parent, const QVariantList &args)
		: Plasma::Applet(parent, args),
		m_svg(this),
		m_user(0),
		m_sys(0),
		m_nice(0),
		m_idle(0),
		m_disk(0){

	this->m_svg.setImagePath("widgets/plot-background");
	setBackgroundHints(DefaultBackground);
	setHasConfigurationInterface(true);
	resize(100, 100);

}
systemMonitor::~systemMonitor(){

	if(hasFailedToLaunch()){

	}else {

	}

}

void systemMonitor::init(){

    readConfig();

	Plasma::DataEngine *cpu_mon = dataEngine("cpumonitor");
	Plasma::DataEngine *mem_mon = dataEngine("memmonitor");
	cpu_mon->connectSource("Average CPU Usage", this, 500);
	mem_mon->connectSource("RAM", this, 500);
	mem_mon->connectSource("Swap", this, 500);

}

void systemMonitor::constraintsEvent(Plasma::Constraints constraints){

	if(constraints.testFlag(Plasma::FormFactorConstraint)){

		if(formFactor() == Plasma::Planar || formFactor() == Plasma::MediaCenter){
			setAspectRatioMode(Plasma::IgnoreAspectRatio);
		}else {
			setAspectRatioMode(Plasma::ConstrainedSquare);
		}

	}

}

void systemMonitor::createConfigurationInterface(KConfigDialog *parent){

	QWidget *widGeneral = new QWidget();
	uiGeneral.setupUi(widGeneral);
	
	uiGeneral.chkIsVertical->setChecked(m_isVertical);

	QWidget *widColours = new QWidget();
	uiColours.setupUi(widColours);

	uiColours.kcbCpuUser->setColor(this->m_cpuUserColour);
	uiColours.kcbCpuNice->setColor(this->m_cpuNiceColour);
	uiColours.kcbCpuDisk->setColor(this->m_cpuDiskColour);
	uiColours.kcbCpuSystem->setColor(this->m_cpuSysColour);
	uiColours.kcbRamBuffers->setColor(this->m_ramBuffersColour);
	uiColours.kcbRamCached->setColor(this->m_ramCachedColour);
	uiColours.kcbRamUser->setColor(this->m_ramUsedColour);
	uiColours.kcbSwap->setColor(this->m_swapUsedColour);

	parent->setButtons (KDialog::Ok | KDialog::Cancel | KDialog::Apply );
	connect(parent, SIGNAL(applyClicked()), this, SLOT(configUpdated()));
	connect(parent, SIGNAL(okClicked()), this, SLOT(configUpdated()));

	parent->addPage(widGeneral, "General", icon(), QString(), false);
	parent->addPage(widColours, "Colours", icon(), QString(), false);
	
	parent->setFaceType(KConfigDialog::Tabbed);

}

void systemMonitor::configUpdated(){

	KConfigGroup cg = config();

	if(uiGeneral.chkIsVertical->isChecked() != this->m_isVertical){
		this->m_isVertical = uiGeneral.chkIsVertical->isChecked();
		cg.writeEntry("vertical", this->m_isVertical);
	}
	if(uiColours.kcbCpuUser->color() != this->m_cpuUserColour){
		this->m_cpuUserColour = uiColours.kcbCpuUser->color();
		cg.writeEntry("colour_cpu_user", this->m_cpuUserColour.name());
	}
	if(uiColours.kcbCpuNice->color() != this->m_cpuNiceColour){
		this->m_cpuNiceColour = uiColours.kcbCpuNice->color();
		cg.writeEntry("colour_cpu_nice", this->m_cpuNiceColour.name());
	}
	if(uiColours.kcbCpuDisk->color() != this->m_cpuDiskColour){
		this->m_cpuDiskColour = uiColours.kcbCpuDisk->color();
		cg.writeEntry("colour_cpu_disk", this->m_cpuDiskColour.name());
	}
	if(uiColours.kcbCpuSystem->color() != this->m_cpuSysColour){
		this->m_cpuSysColour = uiColours.kcbCpuSystem->color();
		cg.writeEntry("colour_cpu_sys", this->m_cpuSysColour.name());
	}
	if(uiColours.kcbRamCached->color() != this->m_ramCachedColour){
		this->m_ramCachedColour = uiColours.kcbRamCached->color();
		cg.writeEntry("colour_ram_cached", this->m_ramCachedColour.name());
	}
	if(uiColours.kcbRamBuffers->color() != this->m_ramBuffersColour){
		this->m_ramBuffersColour = uiColours.kcbRamBuffers->color();
		cg.writeEntry("colour_ram_buffers", this->m_ramBuffersColour.name());
	}
	if(uiColours.kcbRamUser->color() != this->m_ramUsedColour){
		this->m_ramUsedColour = uiColours.kcbRamUser->color();
		cg.writeEntry("colour_ram_used", this->m_ramUsedColour.name());
	}
	if(uiColours.kcbSwap->color() != this->m_swapUsedColour){
		this->m_swapUsedColour = uiColours.kcbSwap->color();
		cg.writeEntry("colour_swap_used", this->m_swapUsedColour.name());
	}

	emit configNeedsSaving();
	update();

}

void systemMonitor::dataUpdated(const QString& source, const Plasma::DataEngine::Data &data){

	if(source == "0" || source == "Average CPU Usage"){
	
		m_user = data["User"].toDouble();
		m_sys = data["Sys"].toDouble();
		m_nice = data["Nice"].toDouble();
		m_idle = data["Idle"].toDouble();
		m_disk = data["Disk"].toDouble();

	}else if(source == "RAM"){

		m_ramfree = data["Free"].toDouble();
		m_ramused = data["Used"].toDouble();
		m_rambuffers = data["Buffers"].toDouble();
		m_ramcached = data["Cached"].toDouble();		

	}else if(source == "Swap"){

		m_swapfree = data["Free"].toDouble();
		m_swapused = data["Used"].toDouble();

	}


	update();

}

void systemMonitor::paintCPUUsage(QPainter *p, const QStyleOptionGraphicsItem *option, const QRect& contentsRect){

	Q_UNUSED(option)

	p->save();

	if (this->m_isVertical)
	{
		p->translate(0, contentsRect.height() * m_idle);
	}
	else
	{
		p->translate(contentsRect.width() * (1-m_idle-m_user), 0);
	}

	p->setBrush(this->m_cpuUserColour);
	p->setPen(this->m_cpuUserColour);

	if (this->m_isVertical)
	{
	p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * 0.33, contentsRect.height() * m_user));
	p->translate(0, contentsRect.height() * m_user);
	}
	else
	{
	p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * m_user, contentsRect.height() * 0.33));
	p->translate(contentsRect.width() * m_nice * -1, 0);
	}

	p->setBrush(this->m_cpuNiceColour);
	p->setPen(this->m_cpuNiceColour);

	if (this->m_isVertical)
	{
	p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * 0.33, contentsRect.height() * m_nice));
	p->translate(0, contentsRect.height() * m_nice);
	}
	else
	{
	p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * m_nice, contentsRect.height() * 0.33));
	p->translate(contentsRect.width() * m_disk * -1, 0);
	}

	p->setBrush(this->m_cpuDiskColour);
	p->setPen(this->m_cpuDiskColour);

	if (this->m_isVertical){
	p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * 0.33, contentsRect.height() * m_disk));
	p->translate(0, contentsRect.height() * m_disk);
	}
	else
	{
	p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * m_disk, contentsRect.height() * 0.33));
	p->translate(contentsRect.width() * m_sys * -1, 0);
	}

	p->setBrush(this->m_cpuSysColour);
	p->setPen(this->m_cpuSysColour);

	if (this->m_isVertical)
	{
		p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * 0.33, contentsRect.height() * m_sys));
	}
	else
	{
		p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * m_sys, contentsRect.height() * 0.33));
	}

	p->restore();

}

void systemMonitor::paintSwapUsage(QPainter *p, const QStyleOptionGraphicsItem *option, const QRect& contentsRect){

	Q_UNUSED(option)

	p->save();

	if (this->m_isVertical)
		p->translate(contentsRect.width() * 0.66, contentsRect.height() * m_swapfree);
	else
		p->translate(0, contentsRect.height() * 0.66);

	p->setBrush(this->m_swapUsedColour);
	p->setPen(this->m_swapUsedColour);

	if (this->m_isVertical)
	{
		p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * 0.33, contentsRect.height() * m_swapused));
	}
	else
	{
		p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * m_swapused, contentsRect.height() * 0.33));
	}

	p->restore();

}

void systemMonitor::paintRAMUsage(QPainter *p, const QStyleOptionGraphicsItem *option, const QRect& contentsRect){

	Q_UNUSED(option)

        p->save();

	if (this->m_isVertical)
	{
		p->translate(contentsRect.width() * 0.33, contentsRect.height() * m_ramfree);
	}
	else
	{
		p->translate(contentsRect.width() * (1-m_ramfree-m_ramcached), contentsRect.height() * 0.33);
	}

	p->setBrush(this->m_ramCachedColour);
	p->setPen(this->m_ramCachedColour);

	if (this->m_isVertical)
	{
		p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * 0.33, contentsRect.height() * m_ramcached));
		p->translate(0, contentsRect.height() * m_ramcached);
	}
	else
	{
		p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * m_ramcached, contentsRect.height() * 0.33 ));
		p->translate(contentsRect.width() * m_rambuffers * -1, 0);
	}

	p->setBrush(this->m_ramBuffersColour);
	p->setPen(this->m_ramBuffersColour);

	if (this->m_isVertical)
	{
		p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * 0.33, contentsRect.height() * m_rambuffers));
		p->translate(0, contentsRect.height() * m_rambuffers);
	}
	else
	{
		p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * m_rambuffers, contentsRect.height() * 	0.33));
		p->translate(contentsRect.width() * m_ramused * -1, 0);
	}

	p->setBrush(this->m_ramUsedColour);
	p->setPen(this->m_ramUsedColour);
	if (this->m_isVertical)
	{
		p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * 0.33, contentsRect.height() * m_ramused));
	}
	else
	{
		p->drawRect(QRectF(contentsRect.left(), contentsRect.top(), contentsRect.width() * m_ramused, contentsRect.height() * 0.33));
	}

	p->restore();

}

void systemMonitor::paintInterface(QPainter *p, const QStyleOptionGraphicsItem *option, const QRect& contentsRect){

	Q_UNUSED(option);

	p->setRenderHint(QPainter::SmoothPixmapTransform);
	p->setRenderHint(QPainter::Antialiasing);

	m_svg.resize((int)contentsRect.width(), (int)contentsRect.height());
	m_svg.paint(p, (int)contentsRect.left(), (int)contentsRect.top());

	paintCPUUsage(p, option, contentsRect);
	paintRAMUsage(p, option, contentsRect);
	paintSwapUsage(p, option, contentsRect);

}

void systemMonitor::readConfig(){

	KConfigGroup cg = config();

	this->m_isVertical = cg.readEntry("vertical", true);
	this->m_cpuUserColour = QColor( cg.readEntry("colour_cpu_user", QString("#0000FF")) );
	this->m_cpuNiceColour = QColor( cg.readEntry("colour_cpu_nice", QString("#FFFF00")) );
	this->m_cpuDiskColour = QColor( cg.readEntry("colour_cpu_disk", QString("#006400")) );
	this->m_cpuSysColour = QColor( cg.readEntry("colour_cpu_sys", QString("#FF0000")) );
	this->m_ramCachedColour = QColor( cg.readEntry("colour_ram_cached", QString("#007800")) );
	this->m_ramBuffersColour = QColor( cg.readEntry("colour_ram_buffers", QString("#FFFF00")) );
	this->m_ramUsedColour = QColor( cg.readEntry("colour_ram_used", QString("#0000B1")) );
	this->m_swapUsedColour = QColor( cg.readEntry("colour_swap_used", QString("#00CDCD")) );

}


#include "system_monitor.moc"
