/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file sdl_sound.cpp
 * \brief Implementation of the bear::audio::sdl_sound class.
 * \author Julien Jorge
 */
#include "audio/sdl_sound.hpp"
#include "audio/sound_manager.hpp"

#include <SDL/SDL.h>
#include <claw/assert.hpp>
#include <claw/exception.hpp>
#include <claw/logger.hpp>
#include <cmath>
#include <limits>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \post is_empty() == true.
 */
bear::audio::sdl_sound::channel_attribute::channel_attribute()
  : m_sound(NULL)
{

} // sdl_sound::channel_attribute::channel_attribute()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::audio::sdl_sound::channel_attribute::~channel_attribute()
{
  clear();
} // sdl_sound::channel_attribute::~channel_attribute()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the sound attribute.
 * \param s The sound.
 * \post is_empty() = false.
 */
void bear::audio::sdl_sound::channel_attribute::set_sound( const sdl_sound& s )
{
  m_sound = &s;
} // sdl_sound::channel_attribute::set_sound()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the stored sound.
 * \pre is_empty() = false.
 */
const bear::audio::sdl_sound&
bear::audio::sdl_sound::channel_attribute::get_sound() const
{
  CLAW_PRECOND( m_sound != NULL );

  return *m_sound;
} // sdl_sound::channel_attribute::get_sound()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the sound.
 * \param p The position.
 */
void bear::audio::sdl_sound::channel_attribute::set_position
( const claw::math::coordinate_2d<int>& p )
{
  m_position = p;
} // sdl_sound::channel_attribute::set_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position of the sound.
 */
const claw::math::coordinate_2d<int>&
bear::audio::sdl_sound::channel_attribute::get_position() const
{
  return m_position;
} // sdl_sound::channel_attribute::get_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove the sound attribute.
 */
void bear::audio::sdl_sound::channel_attribute::clear()
{
  m_sound = NULL;
} // sdl_sound::channel_attribute::clear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there's a sound associated.
 */
bool bear::audio::sdl_sound::channel_attribute::is_empty() const
{
  return m_sound == NULL;
} // sdl_sound::channel_attribute::is_empty()




/*----------------------------------------------------------------------------*/
std::vector<bear::audio::sdl_sound::channel_attribute*>
bear::audio::sdl_sound::s_playing_channels;

unsigned int bear::audio::sdl_sound::s_audio_rate = 22050;
unsigned int bear::audio::sdl_sound::s_audio_format = AUDIO_S16;
unsigned int bear::audio::sdl_sound::s_audio_channels = 2;
unsigned int bear::audio::sdl_sound::s_audio_buffers = 4096;

unsigned int bear::audio::sdl_sound::s_silent_distance = 1000;
unsigned int bear::audio::sdl_sound::s_full_volume_distance = 400;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param file The stream containing the wav file.
 * \param owner The instance of sound_manager who stores me.
 */
bear::audio::sdl_sound::sdl_sound( std::istream& file, sound_manager& owner )
  : sound(owner), m_sound(NULL)
{
  SDL_RWops* rw;

  file.seekg( 0, std::ios::end );
  unsigned int file_size = file.tellg();
  file.seekg( 0, std::ios::beg );

  char* buffer = new char[file_size];
  bool ok = false;

  file.read( buffer, file_size );

  rw = SDL_RWFromMem(buffer, file_size);

  if (rw)
    {
      m_sound = Mix_LoadWAV_RW( rw, 1 );
      ok = (m_sound != NULL);
    }
  delete[] buffer;

  if (!m_sound)
    throw CLAW_EXCEPTION( Mix_GetError() );
} // sdl_sound::sdl_sound()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::audio::sdl_sound::~sdl_sound()
{
  while ( !m_channels.empty() )
    stop();

  Mix_FreeChunk( m_sound );
} // sdl_sound::~sdl_sound()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start to play the sound.
 */
void bear::audio::sdl_sound::play()
{
  inside_play(0);
} // sdl_sound::play()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start to play the sound, with an effect.
 * \param effect The effect applied to the sound.
 */
void bear::audio::sdl_sound::play( const sound_effect& effect )
{
  int channel = inside_play( effect.get_loops() );

  if ( channel != -1)
    Mix_Volume( channel, (int)(effect.get_volume() * MIX_MAX_VOLUME) );
} // sdl_sound::play() [effect]

/*----------------------------------------------------------------------------*/
/**
 * \brief Start to play the sound, giving a position for the sound.
 * \param position The position of this sound.
 */
void
bear::audio::sdl_sound::play( const claw::math::coordinate_2d<int>& position )
{
  sound_effect effect;
  int channel = inside_play(0);

  if ( channel != -1)
    set_channel_position(channel, position);
} // sdl_sound::play() [position]

/*----------------------------------------------------------------------------*/
/**
 * \brief Start to play the sound, with an effect.
 * \param position The position of this sound.
 * \param effect The effect applied to the sound.
 */
void bear::audio::sdl_sound::play
( const claw::math::coordinate_2d<int>& position, const sound_effect& effect )
{
  int channel = inside_play( effect.get_loops() );

  if ( channel != -1)
    {
      SDL_LockAudio();
      Mix_Volume( channel, (int)(effect.get_volume() * MIX_MAX_VOLUME) );
      set_channel_position(channel, position);
      SDL_UnlockAudio();
    }
} // sdl_sound::play() [position, effect]

/*----------------------------------------------------------------------------*/
/**
 * \brief Stop the sound.
 */
void bear::audio::sdl_sound::stop()
{
  if (!m_channels.empty())
    {
      int channel = m_channels.front();
      m_channels.pop_front();

      Mix_HaltChannel( channel );
    }
  else
    claw::logger << claw::log_warning << "sdl_sound::stop(): No open channel."
                 << claw::lendl;
} // sdl_sound::stop()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the SDL.
 */
bool bear::audio::sdl_sound::initialize()
{
  bool result = false;

  if ( SDL_InitSubSystem(SDL_INIT_AUDIO) != 0 )
    claw::logger << claw::log_error << SDL_GetError() << claw::lendl;
  else if ( Mix_OpenAudio(s_audio_rate, s_audio_format, s_audio_channels,
                          s_audio_buffers) != 0 )
    claw::logger << claw::log_error << Mix_GetError() << claw::lendl;
  else
    {
      result = true;
      Mix_ChannelFinished(channel_finished);
    }

  return result;
} // sdl_sound::initialize()

/*----------------------------------------------------------------------------*/
/**
 * \brief Close the SDL.
 */
void bear::audio::sdl_sound::release()
{
  SDL_QuitSubSystem(SDL_INIT_AUDIO);
} // sdl_sound::release()

/*----------------------------------------------------------------------------*/
/**
 * \brief Callback function called when a channel is finished.
 * \param channel The finished channel.
 */
void bear::audio::sdl_sound::channel_finished(int channel)
{
  /* This is the last use of s_playing_channels[channel] for this sound.
     sdl_sound::finished() can't be const for now because it needs to remove
     channel from the m_channels list. That's why we exceptionally remove the
     const attribute. */
  sdl_sound* s = const_cast<sdl_sound*>
    (&s_playing_channels[channel]->get_sound());

  s->finished(channel);
} // sdl_sound::channel_finished()

/*----------------------------------------------------------------------------*/
/**
 * \brief Callback function appluing a distance tone down effect to a channel.
 * \param channel The channel receiving the effect.
 * \param stream (in/out) Sound data.
 * \param length The size of the stream.
 * \param attr (in) Channel attribute.
 * \pre attr != NULL.
 */
void bear::audio::sdl_sound::distance_tone_down
(int channel, void* stream, int length, void* attr)
{
  CLAW_PRECOND( attr != NULL );

  double tone_down = 1.0;
  char* buffer = static_cast<char*>(stream);
  const channel_attribute* attribute = static_cast<channel_attribute*>(attr);

  const claw::math::coordinate_2d<int> ears =
    attribute->get_sound().get_manager().get_ears_position();

  const claw::math::coordinate_2d<int> pos = attribute->get_position();

  unsigned int distance = abs(ears.x - pos.x) + abs(ears.y - pos.y);

  if ( distance >= s_silent_distance )
    tone_down = 0;
  else if ( distance > s_full_volume_distance )
    tone_down = 1.0 - (float)(distance - s_full_volume_distance)
      / (float)(s_silent_distance - s_full_volume_distance);

  if ( tone_down <= std::numeric_limits<float>::epsilon() )
    std::fill( buffer, buffer + (unsigned int)length, 0 );
  else if ( tone_down < 1.0 )
    for (unsigned int i=0; i!=(unsigned)length; ++i)
      buffer[i] = (char)((float)buffer[i] * tone_down);

} // sdl_sound::distance_tone_down()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start to play the sound.
 * \param loops Number of loops, -1 for infinite.
 * \return The channel in which the sound is played.
 * \pre loops >= -1.
 */
int bear::audio::sdl_sound::inside_play( int loops )
{
  CLAW_PRECOND( loops >= -1 );

  int channel = Mix_PlayChannel(-1, m_sound, loops);

  if (channel == -1)
    claw::logger << claw::log_warning << "sdl_sound::inside_play(): "
                 << Mix_GetError() << claw::lendl;
  else
    {
      SDL_LockAudio();
      m_channels.push_back(channel);
      global_add_channel(channel);
      SDL_UnlockAudio();
    }

  return channel;
} // sdl_sound::inside_play()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position associated to a channel.
 * \param channel The concerned channel.
 * \param position The position of this sound.
 * \pre channel >= 0
 * \pre s_playing_channels[channel]->is_empty() == false.
 * \post s_playing_channels[channel]->get_position() == position.
 */
void bear::audio::sdl_sound::set_channel_position
( int channel, const claw::math::coordinate_2d<int>& position)
{
  CLAW_PRECOND( channel >= 0 );
  CLAW_PRECOND( s_playing_channels[channel]->is_empty() == false );

  s_playing_channels[channel]->set_position( position );

  int ok;
  ok = Mix_RegisterEffect( channel, distance_tone_down, NULL,
                           s_playing_channels[channel] );

  if (!ok)
    claw::logger << claw::log_warning << "sdl_sound::play(): " << Mix_GetError()
                 << claw::lendl;
} // sdl_sound::set_channel_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Associate the current instance to a channel in the common list.
 * \param channel The channel of this instance.
 * \pre channel >= 0
 * \pre s_playing_channels[channel].is_empty().
 * \post sdl_sound::s_playing_channels[channel].get_sound() == *this.
 */
void bear::audio::sdl_sound::global_add_channel( int channel )
{
  CLAW_PRECOND( channel >= 0 );

  if ( (unsigned int)channel >= s_playing_channels.size() )
    s_playing_channels.resize(channel+1, NULL);
  else
    {
      CLAW_PRECOND( s_playing_channels[channel] == NULL );
    }

  s_playing_channels[channel] = new channel_attribute;
  s_playing_channels[channel]->set_sound(*this);
} // sdl_sound::global_add_channel()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove a finished channel from the playing list.
 * \brief channel The channel to remove.
 * \pre channel >= 0
 * \post sdl_sound::s_playing_channels[channel]->is_empty() = true.
 * \post m_channels.find( channel ) == m_channels.end().
 */
void bear::audio::sdl_sound::finished( int channel )
{
  CLAW_PRECOND( channel >= 0 );

  if ( !Mix_UnregisterAllEffects(channel) )
    claw::logger << claw::log_warning << "sdl_sound::finished(): "
                 << Mix_GetError() << claw::lendl;

  std::list<int>::iterator it;
  bool ok = false;

  it = m_channels.begin();
  while ( !ok && (it != m_channels.end()) )
    if ( *it == channel )
      ok = true;
    else
      ++it;

  if (ok)
    m_channels.erase( it );
  else
    claw::logger << claw::log_warning << "sdl_sound::finished():channel "
                 << channel << " is not in the list" << claw::lendl;

  delete s_playing_channels[channel];
  s_playing_channels[channel] = NULL;
} // sdl_sound::finished()
