/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file item_handle.cpp
 * \brief Implementation of the bear::engine::item_handle class.
 * \author Julien Jorge
 */
#include "engine/item_handle.hpp"

#include "engine/base_item.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::engine::item_handle::item_handle()
  : m_item(NULL)
{

} // item_handle::item_handle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param item The item to handle.
 */
bear::engine::item_handle::item_handle( base_item* item )
  : m_item(item)
{
  if ( m_item )
    m_item->add_handle( this );
} // item_handle::item_handle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 * \param that The handle from which we copy.
 */
bear::engine::item_handle::item_handle( const item_handle& that )
  : m_item(that.m_item)
{
  if ( m_item )
    m_item->add_handle( this );
} // item_handle::item_handle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::engine::item_handle::~item_handle()
{
  if ( m_item )
    m_item->remove_handle( this );
} // item_handle::item_handle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the handle point to a valid entity.
 */
bool bear::engine::item_handle::is_valid()
{
  return get() != NULL;
} // item_handle::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the pointed entity.
 */
bear::engine::base_item* bear::engine::item_handle::get()
{
  return m_item;
} // item_handle::get()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the pointed entity.
 */
const bear::engine::base_item* bear::engine::item_handle::get() const
{
  return m_item;
} // item_handle::get()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the handle point to a valid entity.
 * \return is_valid()
 */
bear::engine::item_handle::operator bool()
{
  return is_valid();
} // item_handle::operator bool()

/*----------------------------------------------------------------------------*/
/**
 * \brief Cast to the pointer type.
 * \return get()
 */
bear::engine::item_handle::operator base_item*()
{
  return get();
} // item_handle::operator base_item*()

/*----------------------------------------------------------------------------*/
/**
 * \brief Cast to the pointer type.
 * \return get()
 */
bear::engine::item_handle::operator const base_item*() const
{
  return get();
} // item_handle::operator const base_item*()

/*----------------------------------------------------------------------------*/
/**
 * \brief Pointer-to-member operator.
 * \return get()
 */
bear::engine::base_item* bear::engine::item_handle::operator->()
{
  return get();
} // item_handle::operator->()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assigment operator.
 * \param item The item to assign.
 */
bear::engine::item_handle&
bear::engine::item_handle::operator=( base_item* item )
{
  if ( m_item != NULL )
    m_item->remove_handle( this );

  m_item = item;

  if ( m_item != NULL )
    m_item->add_handle( this );
    
  return *this;
} // item_handle::operator=()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assigment operator.
 * \param that The item to assign.
 */
bear::engine::item_handle&
bear::engine::item_handle::operator=( const bear::engine::item_handle& that )
{
  if ( m_item != NULL )
    m_item->remove_handle( this );

  m_item = that.m_item;

  if ( m_item )
    m_item->add_handle( this );

  return *this;
} // item_handle::operator=()
