/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file level.cpp
 * \brief Implementation of the bear::engine::level class.
 * \author Julien Jorge
 */
#include "engine/level.hpp"

#include <algorithm>
#include <claw/functional.hpp>
#include "engine/camera.hpp"
#include "engine/game.hpp"
#include "engine/level_loader.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param f The file from which we will read the level.
 * \param name The name of the level.
 */
bear::engine::level::screen_effect::screen_effect
( const std::string& _name, unsigned int _layer_index,
  visual::screen_effect* _effect )
  : name(_name), layer_index(_layer_index), effect(_effect)
{

} // level::screen_effect::screen_effect()




/*----------------------------------------------------------------------------*/
// arbitrary value
const unsigned int bear::engine::level::s_level_margin = 100;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param f The file from which we will read the level.
 * \param name The name of the level.
 */
bear::engine::level::level( compiled_file& f, const std::string& name )
  : m_name(name), m_camera(NULL), m_level_globals(NULL)
{
  level_loader ldr(f);

  try
    {
      ldr.complete_run();
      initialise_from_loader(ldr);
    }
  catch( ... )
    {
      clear();
      throw;
    }
} // level::level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param ldr The loader with which the level was loaded.
 * \param name The name of the level.
 */
bear::engine::level::level( level_loader& ldr, const std::string& name )
  : m_name(name), m_camera(NULL), m_level_globals(NULL)
{
  try
    {
      initialise_from_loader(ldr);
    }
  catch( ... )
    {
      clear();
      throw;
    }
} // level::level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::engine::level::~level()
{
  stop();
  clear();
} // level::~level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start the level.
 */
void bear::engine::level::start()
{
  m_camera = new camera( "camera", m_camera_size );

  universe::rectangle_type valid_area
    ( s_level_margin, s_level_margin,
      get_size().x - 2 * s_level_margin, get_size().y - 2 * s_level_margin );

  m_camera->set_valid_area( valid_area );
  m_camera->set_position( claw::math::coordinate_2d<unsigned int>(0, 0) );

  m_level_globals->register_item(*m_camera);

  if ( m_level_globals->music_exists( m_music ) )
    m_level_globals->get_music( m_music ).play();

  for (unsigned int i=0; i!=m_layers.size(); ++i)
    m_layers[i]->start();

  m_camera->initialise_position();
} // level::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Stop the level.
 */
void bear::engine::level::stop()
{
  if ( m_level_globals->music_exists( m_music ) )
    m_level_globals->get_music( m_music ).stop();
} // level::stop()

/*----------------------------------------------------------------------------*/
/**
 * \brief Progress all layers in the active area.
 * \param elapsed_time Elapsed time since the last call.
 */
void bear::engine::level::progress( universe::time_type elapsed_time )
{
  region_type active_regions;

  get_active_regions( active_regions );

  for (unsigned int i=0; i!=m_layers.size(); ++i)
    {
      region_type areas(active_regions);
      get_layer_region(i, areas);
      m_layers[i]->progress( areas, elapsed_time );
    }

  m_camera->auto_position();
  m_level_globals->set_ears_position
    ( m_camera->get_center().cast_value_type_to<int>() );
} // level::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the visible part of the level on the screen.
 * \param screen The screen on which we draw.
 */
void bear::engine::level::render( visual::screen& screen ) const
{
  for (unsigned int i=0; i!=m_layers.size(); ++i)
    {
      region_type r;
      add_region( r, m_camera->get_center() );

      universe::rectangle_type active( r.front() );
      std::list<scene_visual> visuals;

      get_layer_area(i, active); // the active area scaled in the layer

      m_layers[i]->get_visual( visuals, active );
      visuals.sort( scene_visual::z_position_compare() );

      universe::rectangle_type area( m_camera->get_focus() );
      get_layer_area(i, area);   // the camera scaled in the layer
      render( visuals, area.position, screen,
              (double)screen.get_size().x / m_camera_size.x,
              (double)screen.get_size().y / m_camera_size.y );

      apply_effects( screen, i );
    }
} // level::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw a part of the action layer on the screen.
 * \param screen The screen on which we draw.
 * \param rect The region of the action layer to render.
 */
void bear::engine::level::shot
( visual::screen& screen, const universe::rectangle_type& rect ) const
{
  universe::rectangle_type active( rect );

  // stay in the level
  if ( active.position.x + active.width >= get_size().x )
    active.width = get_size().x - active.position.x;

  if ( active.position.y + active.height >= get_size().y )
    active.height = get_size().y - active.position.y;

  for (unsigned int i=0; i!=m_layers.size(); ++i)
    if ( m_layers[i]->get_size() == get_size() )
      {
        std::list<scene_visual> visuals;
        m_layers[i]->get_visual( visuals, active );
        visuals.sort( scene_visual::z_position_compare() );

        render( visuals, active.position, screen,
                m_camera->get_focus().width / rect.width,
                m_camera->get_focus().height / rect.height );
      }
} // level::shot()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an effect in the render process.
 * \param name The identifier of your effect.
 * \param layer_index The index of the layer on which it applies
 * \param effect The effect to apply.
 * \pre \a effect is dynamically allocated.
 * \pre layer_index < get_depth()
 * \remark We won't check if the name is already in use or not.
 */
void bear::engine::level::add_effect
( const std::string& name, unsigned int layer_index,
  visual::screen_effect* effect )
{
  CLAW_PRECOND( effect != NULL );
  CLAW_PRECOND( layer_index < get_depth() );

  m_effects.push_front( screen_effect(name, layer_index, effect) );
} // level::add_effect()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an effect just before the status_layer.
 * \param name The identifier of your effect.
 * \param effect The effect to apply.
 * \pre \a effect is dynamically allocated.
 * \pre get_depth > 1
 * \remark This method is equivalent to add_effect( name, get_depth() - 2,
 *         effect )
 */
void bear::engine::level::add_effect
( const std::string& name, visual::screen_effect* effect )
{
  CLAW_PRECOND( effect != NULL );
  CLAW_PRECOND( get_depth() > 1 );

  add_effect( name, get_depth() - 2, effect );
} // level::add_effect()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove an effect from the render process.
 * \param name The name of the effect to remove.
 * \remark We remove the first effect that matches \a name.
 */
void bear::engine::level::remove_effect( const std::string& name )
{
  effect_stack::iterator it = m_effects.begin();
  bool found = false;

  while ( !found && (it != m_effects.end()) )
    if (it->name == name)
      found = true;
    else
      ++it;

  if (found)
    {
      delete it->effect;
      m_effects.erase(it);
    }
  else
    claw::logger << claw::log_warning << "Can't find effect '" << name
                 << "' for removing." << claw::lendl;
} // level::remove_effect()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the size of the level.
 */
const bear::universe::size_box_type& bear::engine::level::get_size() const
{
  return m_level_size;
} // level::get_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the number of layers.
 */
unsigned int bear::engine::level::get_depth() const
{
  return m_layers.size();
} // level::get_depth()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the level.
 */
const std::string& bear::engine::level::get_name() const
{
  return m_name;
} // level::get_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the globals resources of the level.
 */
bear::engine::level_globals& bear::engine::level::get_globals()
{
  return *m_level_globals;
} // level::get_globals()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialise the level with the data of a loader.
 * \param ldr The loader from which we take the level.
 */
void bear::engine::level::initialise_from_loader( level_loader& ldr )
{
  m_level_size = ldr.get_level_size();
  m_camera_size = ldr.get_camera_size();
  m_music = ldr.get_level_music();

  ldr.drop_level_content( m_layers, m_level_globals );
} // level::initialise_from_loader()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the visible sprites.
 * \param visuals The sprites to display and their coordinates in the world.
 * \param cam_pos The position of the camera.
 * \param screen The screen on which we draw.
 * \param r_w Ratio on the width of the sprites.
 * \param r_h Ratio on the height of the sprites.
 */
void bear::engine::level::render
( const std::list<scene_visual>& visuals,
  const universe::position_type& cam_pos, visual::screen& screen,
  double r_w, double r_h ) const
{
  std::list<scene_visual>::const_iterator it;

  for ( it=visuals.begin(); it!=visuals.end(); ++it )
    {
      universe::position_type pos(it->position - cam_pos);
      visual::sprite spr( it->sprite );

      pos.x *= r_w;
      pos.y *= r_h;
      spr.set_size
        ( (unsigned int)
          ( (double)it->sprite.width() * r_w + pos.x - (int)pos.x ),
          (unsigned int)
          ( (double)it->sprite.height() * r_h + pos.y - (int)pos.y ) );

      screen.render( pos.cast_value_type_to<int>(), spr, it->angle );
    }
} // level::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply an effect on the screen.
 * \param screen The screen on which we draw the effect.
 * \param layer_index The index of the layer on which we apply the effects.
 */
void bear::engine::level::apply_effects
( visual::screen& screen, unsigned int layer_index ) const
{
  effect_stack::const_iterator it;
  bool apply=false;

  // effects are costful, so be sure we have to apply one before calling the
  // screen.{begin,end}_effects() methods.

  for ( it = m_effects.begin(); !apply && (it!=m_effects.end()); )
    if (it->layer_index == layer_index)
      apply = true;
    else
      ++it;

  if (apply)
    {
      screen.begin_effects();

      for ( ; it!=m_effects.end(); ++it )
        if (it->layer_index == layer_index)
          screen.apply_effect( *it->effect );

      screen.end_effects();
    }
} // level::apply_effect()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove all the layers.
 */
void bear::engine::level::clear()
{
  std::for_each( m_layers.begin(), m_layers.end(),
                 claw::delete_function<layer*>() );

  effect_stack::const_iterator it;

  for ( it = m_effects.begin(); it!=m_effects.end(); ++it )
    delete it->effect;

  m_layers.clear();

  if ( m_camera != NULL )
    delete m_camera;

  if ( m_level_globals != NULL )
    delete m_level_globals;
} // layer::clear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the repositioned region in a layer from the area in the
 *        level/action layer.
 * \param layer_index The index of the layer in which we want the repositioned
 *        region.
 * \param the_region (in) The region in the action layer, (out) the repositioned
 *        region.
 */
void bear::engine::level::get_layer_region
( unsigned int layer_index, region_type& the_region ) const
{
  region_type::iterator it;

  for (it=the_region.begin(); it!=the_region.end(); ++it)
    get_layer_area( layer_index, *it );
} // level::get_layer_region()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the repositioned area in a layer from the area in the level/action
 *        layer.
 * \param layer_index The index of the layer in which we want the repositioned
 *        area.
 * \param area (in) The area in the level, (out) the repositioned area.
 */
void bear::engine::level::get_layer_area
( unsigned int layer_index, universe::rectangle_type& area ) const
{
  if ( area.width > m_layers[layer_index]->get_size().x )
    area.position.x = 0;
  else if ( m_level_size.x > area.width )
    area.position.x =
      ( area.position.x * ( m_layers[layer_index]->get_size().x - area.width ) )
      / ( m_level_size.x - area.width );

  if ( area.height > m_layers[layer_index]->get_size().y )
    area.position.y = 0;
  else if ( m_level_size.y > area.height )
    area.position.y =
      ( area.position.y *
        ( m_layers[layer_index]->get_size().y - area.height )
        ) / ( m_level_size.y - area.height );
} // level::get_layer_area()

/*----------------------------------------------------------------------------*/
/**
 * \brief Find the active regions in the level.
 * \param active_regions (out) The active regions in the level.
 */
void
bear::engine::level::get_active_regions( region_type& active_regions ) const
{
  player::get_instance_message msg;

  unsigned int player_index = 1;

  while (m_level_globals->send_message(player::player_name(player_index), msg))
    {
      universe::position_type region =
        msg.get_instance()->get_center_of_mass();
      add_region( active_regions, region );
      ++player_index;
    }

  add_region( active_regions, m_camera->get_center() );
} // level::get_active_regions()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an active region centered on a point in a list of regions.
 * \param active_regions (out) The active regions.
 * \param center The point on which we center the region.
 */
void
bear::engine::level::add_region
( region_type& active_regions, const universe::position_type& center ) const
{
  universe::size_box_type box_size( m_camera_size );
  box_size *= game::get_instance().get_active_area_ratio();
  universe::rectangle_type box;

  // set top-left position so we stay in the level
  if (center.x >= box_size.x / 2)
    box.position.x = center.x - box_size.x / 2;
  else
    {
      box.position.x = 0;
      box_size.x -= box_size.x / 2 - center.x;
    }

  if (center.y >= box_size.y / 2)
    box.position.y = center.y - box_size.y / 2;
  else
    {
      box.position.y = 0;
      box_size.y -= box_size.y / 2 - center.y;
    }

  // set size for the same reason
  if (box.position.x + box_size.x >= get_size().x)
    box_size.x = get_size().x - box.position.x;

  if (box.position.y + box_size.y >= get_size().y)
    box_size.y = get_size().y - box.position.y;

  box.width = box_size.x;
  box.height = box_size.y;

  active_regions.push_front( box );
} // level::add_region()
