/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file model.tpp
 * \brief Implementation of the bear::engine::model class.
 * \author Julien Jorge
 */

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
template<class Base>
bear::engine::model<Base>::model()
  : m_actions(NULL), m_current_action(NULL),
    m_animation_alignment(align_center_on_center)
{

} // model::model()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
template<class Base>
bear::engine::model<Base>::~model()
{
  if (m_actions)
    delete m_actions;
} // model::~model()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is correctly initialized.
 */
template<class Base>
bool bear::engine::model<Base>::is_valid() const
{
  return /*(m_actions != NULL) &&*/ super::is_valid();
} // model::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Go to the next frame of the animation.
 */
template<class Base>
void bear::engine::model<Base>::progress( universe::time_type elapsed_time )
{
  CLAW_PRECOND( m_current_action );

  next_animation();

  universe::speed_type speed( this->get_speed() );

  if ( (speed.x < 0) && !m_current_action->animation->is_mirrored() )
    m_current_action->animation->mirror(true);
  else if ( (speed.x > 0) && m_current_action->animation->is_mirrored() )
    m_current_action->animation->mirror(false);
} // model::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite representing the item.
 * \param visuals (out) The sprites of the item, and their positions.
 */
template<class Base>
void
bear::engine::model<Base>::get_visual( std::list<scene_visual>& visuals ) const
{
  CLAW_PRECOND( m_current_action );

  visuals.push_front
    ( scene_visual
      ( this->get_position() + this->get_gap(),
        m_current_action->animation->get_sprite(),
        this->get_angle(), this->get_z_position() ) );
} // model::get_visual()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an animation of the model.
 * \param action_name The action from which we want the animation.
 */
template<class Base>
const bear::visual::animation&
bear::engine::model<Base>::get_animation( const std::string& action_name ) const
{
  return *m_actions->get_action(action_name).animation;	
} // model::get_animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the sprite of the animation.
 */
template<class Base>
void bear::engine::model<Base>::mirror(bool b)
{
  m_current_action->animation->mirror(b);
} // model::mirror()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the flip of the animation.
 */
template<class Base>
void bear::engine::model<Base>::flip(bool b)
{
  m_current_action->animation->flip(b);
} // model::flip()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the alpha transparence of the animation.
 */
template<class Base>
void bear::engine::model<Base>::set_alpha_blend(double alpha)
{
  m_current_action->animation->set_alpha_blend(alpha);
} // model::set_alpha_blend()


/*----------------------------------------------------------------------------*/
/**
 * \brief Pass at the next iteration.
 */
template<class Base>
void bear::engine::model<Base>::next_animation()
{
  m_current_action->animation->next();
} // model::next_animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initiate the animation.
 */
template<class Base>
void bear::engine::model<Base>::reset_animation()
{
  return m_current_action->animation->reset();
} // model::reset_animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Indicates if the iteration is finish.
 */
template<class Base>
bool bear::engine::model<Base>::is_finished() const
{
  return m_current_action->animation->is_finished();
} // model::is_finished()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the model file.
 * \param file_name The path to the model file.
 */
template<class Base>
void bear::engine::model<Base>::set_model( const std::string& file_name )
{
  CLAW_PRECOND( m_actions == NULL );

  m_actions = new model_file( file_name );
} // model::set_model()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the alignment of the animation and the bounding box.
 * \param align Tell how we adjust the animation and the bounding box.
 */
template<class Base>
void
bear::engine::model<Base>::set_animation_alignment( animation_alignment align )
{
  m_animation_alignment = align;
} // model::set_animation_alignment()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start a new action.
 * \param name The name of the new action.
 * \pre There is an action called \a name.
 */
template<class Base>
void bear::engine::model<Base>::execute_action( const std::string& name )
{
  m_current_action = &m_actions->get_action(name);
  compute_gap();

  if (m_current_action->sound)
    m_current_action->sound->play();
} // model::execute_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start a new action.
 * \param name The name of the new action.
 * \pre There is an action called \a name.
 */
template<class Base>
void bear::engine::model<Base>::compute_gap()
{
  universe::coordinate_type x;
  universe::coordinate_type y;

  const claw::math::coordinate_2d<int> anim_size
    ( m_current_action->animation->get_max_size() );

  switch( m_animation_alignment )
    {
    case align_center_on_bottom:
    case align_center_on_top:
    case align_center_on_center:
      x = (this->get_width() - anim_size.x) / 2;
      break;
    case align_center_on_left:
    case align_top_left_on_top_left:
    case align_bottom_left_on_bottom_left:
      x = 0;
      break;
    case align_center_on_right:
    case align_top_right_on_top_right:
    case align_bottom_right_on_bottom_right:
      x = this->get_width() - anim_size.x;
      break;
    }

  switch( m_animation_alignment )
    {
    case align_center_on_bottom:
    case align_bottom_left_on_bottom_left:
    case align_bottom_right_on_bottom_right:
      y = this->get_height() - anim_size.y;
      break;
    case align_center_on_left:
    case align_center_on_right:
    case align_center_on_center:
      y = (this->get_height() - anim_size.y) / 2;
      break;
    case align_center_on_top:
    case align_top_left_on_top_left:
    case align_top_right_on_top_right:
      y = 0;
      break;
    }

  this->set_gap( (int)x, (int)y );
} // model::compute_gap()
