/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file model_file.tpp
 * \brief Implementation of the bear::engine::model_file class.
 * \author Julien Jorge
 */
#include "engine/model_file.hpp"

#include <sstream>
#include <claw/assert.hpp>

#include "engine/game.hpp"
#include "engine/resource_pool.hpp"
#include "model_code_value.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param file_name The name of the file describing the model.
 */
bear::engine::model_file::model_file( const std::string& file_name )
{
  std::stringstream source_file;

  resource_pool::get_instance().get_file( file_name, source_file );

  compiled_file f(source_file, true);

  load_model(f);
} // model_file::model_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::engine::model_file::~model_file()
{
  clear();
} // model_file::~model_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an action.
 * \param name The name of the action to get.
 */
bear::engine::model_file::action&
bear::engine::model_file::get_action( const std::string& name )
{
  CLAW_PRECOND( m_actions.find(name) != m_actions.end() );

  return m_actions[name];
} // model_file::get_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove all actions.
 */
void bear::engine::model_file::clear()
{
  action_map::iterator it;

  for (it=m_actions.begin(); it!=m_actions.end(); ++it)
    delete it->second.animation;

  m_actions.clear();
} // model_file::clear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an action to the model.
 * \param name The name of the action.
 * \param animation The animation to play.
 * \param sound_name The sound to play.
 * \pre The action \a name doesn't exist.
 * \pre animation != NULL
 *
 * \remark \a animation will be deleted in the destructor.
 */
void bear::engine::model_file::add_action
( const std::string& name, visual::animation* animation,
  const std::string& sound_name )
{
  CLAW_PRECOND( m_actions.find(name) == m_actions.end() );
  CLAW_PRECOND( animation );

  m_actions[name].animation = animation;

  if ( sound_name.empty() )
    m_actions[name].sound = NULL;
  else
    m_actions[name].sound =
      &game::get_instance().current_level_globals().get_sound( sound_name );
} // model_file::add_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Build the model from a file.
 * \param f The file from which we load the model.
 */
void bear::engine::model_file::load_model( compiled_file& f )
{
  unsigned int next_code;

  f >> next_code;

  while( next_code != model_code_value::eof )
    next_code = load_action(f);
} // model_file::load_model()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the action layer.
 * \param f The file from which we will read the model.
 */
unsigned int bear::engine::model_file::load_action( compiled_file& f )
{
  unsigned int next_code;
  std::string name, sound;

  f >> name;

  visual::animation* anim = load_animation(f);

  f >> sound;

  add_action( name, anim, sound );

  f >> next_code;

  return next_code;
} // model_file::load_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load a sprite.
 * \param f The file from which we will read the model.
 */
bear::visual::sprite*
bear::engine::model_file::load_sprite( compiled_file& f ) const
{
  std::string image_name;
  claw::math::rectangle<unsigned int> clip;
  bool flip, mirror;
  double alpha;

  f >> image_name >> clip.position.x >> clip.position.y >> clip.width
    >> clip.height >> mirror >> flip >> alpha;

  visual::sprite* result;
  const visual::image& resource =
    game::get_instance().current_level_globals().load_image(image_name);

  result = new visual::sprite( resource, clip );

  result->flip( flip );
  result->mirror( mirror );
  result->set_alpha_blend( alpha );

  return result;
} // model_file::load_sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load an animation.
 * \param f The file from which we will read the model.
 */
bear::visual::animation*
bear::engine::model_file::load_animation( compiled_file& f ) const
{
  unsigned int frames_count;

  f >> frames_count;

  std::vector<visual::sprite*> frames(frames_count);

  for ( unsigned int i=0; i!=frames_count; ++i)
    frames[i] = load_sprite(f);

  unsigned int loops;
  bool flip, mirror, loop_back;
  double alpha;

  f >> mirror >> flip >> alpha >> loops >> loop_back;

  visual::animation* result;

  result = new visual::animation( frames );

  result->flip( flip );
  result->mirror( mirror );
  result->set_alpha_blend( alpha );
  result->set_loops( loops );
  result->set_loop_back( loop_back );

  return result;
} // model_file::load_animation()
