/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file resource_pool.cpp
 * \brief Implementation of the bear::engine::resource_pool class.
 * \author Julien Jorge
 */
#include "engine/resource_pool.hpp"

#include <fstream>
#include <claw/exception.hpp>
#include <claw/functional.hpp>
#include <algorithm>
#include <boost/filesystem/convenience.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the instance.
 */
bear::engine::resource_pool& bear::engine::resource_pool::get_instance()
{
  return super::get_instance();
} // resource_pool::get_instance()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a path in which to seek resources.
 * \param path The path to add.
 */
void bear::engine::resource_pool::add_path( const std::string& path )
{
  m_path.push_front(path);
} // resource_pool::add_path()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a file.
 * \param name The path of the file to get.
 * \param os Where we must write the content of the file.
 */
void bear::engine::resource_pool::get_file
( const std::string& name, std::ostream& os )
{
  std::ifstream f;

  if ( find_file(name, f) )
    {
      f >> os.rdbuf();
      f.close();
    }
  else
    throw CLAW_EXCEPTION( "Can't find file '" + name + "'" );

} // resource_pool::get_file()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if we know a file with a given name.
 * \param name The name of the file to find.
 */
bool bear::engine::resource_pool::exists( const std::string& name ) const
{
  bool result = false;

  std::ifstream f;

  if ( find_file(name, f) )
    {
      f.close();
      result = true;
    }

  return result;
} // resource_pool::exists()

/*----------------------------------------------------------------------------*/
/**
 * \brief Find a file in the paths.
 * \param name The name of the file to find.
 * \param f We will open the file with this variable.
 * \return True if we found the file (f is opened) ; false otherwise.
 */
bool bear::engine::resource_pool::find_file
( const std::string& name, std::ifstream& f ) const
{
  std::list<std::string>::const_iterator it;

  for (it=m_path.begin(); (it!=m_path.end()) && !f.is_open(); ++it)
    {
      const std::string filepath = *it + name;
      const boost::filesystem::path path( filepath, boost::filesystem::native );

      if ( boost::filesystem::exists( path ) )
        if ( !boost::filesystem::is_directory( path ) )
          f.open( filepath.c_str(), std::ios::binary );
    }

  return f.is_open();
} // resource_pool::find_file()
