/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file level.hpp
 * \brief One level in the game.
 * \author Julien Jorge
 */
#ifndef __ENGINE_LEVEL_HPP__
#define __ENGINE_LEVEL_HPP__

#include <vector>

#include "engine/compiled_file.hpp"
#include "engine/layer/layer.hpp"
#include "engine/level_globals.hpp"
#include "visual/screen.hpp"

#include "engine/class_export.hpp"

namespace bear
{
  namespace engine
  {
    class level_loader;
    class camera;

    /**
     * \brief One level in the game.
     * \author Julien Jorge
     */
    class ENGINE_EXPORT level
    {
    private:
      class screen_effect
      {
      public:
        screen_effect( const std::string& _name, unsigned int _layer_index,
                       visual::screen_effect* _effect );

      public:
        /** \brief The identifier of the effect. */
        const std::string name;

        /** \brief The layer on which it applies. */
        const unsigned int layer_index;

        /** \brief The effect to apply. */
        visual::screen_effect* effect;

      }; // class screen_effect

      typedef std::list<screen_effect> effect_stack;

      typedef layer::region_type region_type;

    public:
      level( compiled_file& f, const std::string& name );
      level( level_loader& ldr, const std::string& name );
      ~level();

      void start();
      void stop();

      void progress( universe::time_type elapsed_time );
      void render( visual::screen& screen ) const;

      void shot
      ( visual::screen& screen, const universe::rectangle_type& rect ) const;

      void add_effect
      ( const std::string& name, unsigned int layer_index,
        visual::screen_effect* effect );
      void add_effect
      ( const std::string& name, visual::screen_effect* effect );
      void remove_effect( const std::string& name );

      const universe::size_box_type& get_size() const;
      unsigned int get_depth() const;
      const std::string& get_name() const;
      level_globals& get_globals();

    private:
      void initialise_from_loader( level_loader& ldr );

      void render
      ( const std::list<scene_visual>& visuals,
        const universe::position_type& cam_pos, visual::screen& screen,
        double r_w, double r_h ) const;

      void apply_effects
      ( visual::screen& screen, unsigned int layer_index ) const;

      void clear();

      void get_layer_region
      ( unsigned int layer_index, region_type& the_region ) const;
      void get_layer_area
      ( unsigned int layer_index, universe::rectangle_type& area ) const;

      void get_active_regions( region_type& active_regions ) const;

      void add_region
      ( region_type& active_regions,
        const universe::position_type& center ) const;

    public:
      /** \brief The size of the margins around the level. */
      static const unsigned int s_level_margin;

    private:
      /** \brief The name of the level. */
      const std::string m_name;

      /** \brief Visible/active part of the level. */
      camera* m_camera;

      /** \brief The layers of the level, from the back to the front. */
      std::vector<layer*> m_layers;

      /** \brief The size of the level. */
      universe::size_box_type m_level_size;

      /** \brief The size of the camera. */
      universe::size_box_type m_camera_size;

      /** \brief The effects to apply between the layers. */
      effect_stack m_effects;

      /** \brief Resources of the level. */
      level_globals* m_level_globals;

      /** \brief The default music to play in the level. */
      std::string m_music;

    }; // class level
  } // namespace engine
} // namespace bear

#endif // __ENGINE_LEVEL_HPP__
