/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file model.hpp
 * \brief This class manage some basic informations on items: actions with
 *        their sounds, animations and bounding boxes.
 * \author Julien Jorge
 */
#ifndef __ENGINE_MODEL_HPP__
#define __ENGINE_MODEL_HPP__

#include "engine/model_file.hpp"
#include "engine/basic_renderable_item.hpp"

namespace bear
{
  namespace engine
  {
    /**
     * \brief This class manage some basic informations on items: actions with
     *        their sounds, animations and bounding boxes.
     *
     * \b Template \b arguments:
     *  - \a Base the base class of the model.
     *
     * \remark This class assume that animations are oriented to the right by
     *         default.
     *
     * \author Julien Jorge
     */
    template<class Base>
    class model : public engine::basic_renderable_item<Base>
    {
    public:
      /** \brief The type of the parent class. */
      typedef engine::basic_renderable_item<Base> super;

      /** \brief Tell how we adjust the animation and the bounding box. */
      enum animation_alignment
        {
          align_center_on_bottom,
          align_center_on_left,
          align_center_on_top,
          align_center_on_right,
          align_center_on_center,
          align_top_left_on_top_left,
          align_top_right_on_top_right,
          align_bottom_left_on_bottom_left,
          align_bottom_right_on_bottom_right
        }; // enum animation_alignment

    public:
      model();
      ~model();

      bool is_valid() const;

      void progress( universe::time_type elapsed_time );
      void get_visual( std::list<scene_visual>& visuals ) const;

      void flip(bool b);
      void mirror(bool b);
      void set_alpha_blend(double alpha);
      void next_animation();
      void reset_animation();
      bool is_finished() const;

      const visual::animation&
      get_animation( const std::string& action_name ) const;

    protected:
      void set_model( const std::string& model_file );
      void set_animation_alignment( animation_alignment align );
      void execute_action( const std::string& name );

      void compute_gap();

    private:
      /** \brief The set of actions. */
      model_file* m_actions;

      /** \brief The action we are currently playing. */
      model_file::action* m_current_action;

      /** \brief Tell how we adjust the animation and the bounding box. */
      animation_alignment m_animation_alignment;

    }; // class model
  } // namespace engine
} // namespace bear

#include "engine/code/model.tpp"

#endif // __ENGINE_MODEL_HPP__
