/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file model.hpp
 * \brief This class describes a model : a set of animations and sounds
 *        associated with an action.
 * \author Julien Jorge
 */
#ifndef __ENGINE_MODEL_FILE_HPP__
#define __ENGINE_MODEL_FILE_HPP__

#include "engine/class_export.hpp"
#include "engine/compiled_file.hpp"
#include "visual/animation.hpp"
#include "visual/screen.hpp"
#include "audio/sound.hpp"

namespace bear
{
  namespace engine
  {
    /**
     * \brief This class describes a model: a set of animations and sounds
     *        associated with an action.
     * \author Julien Jorge
     */
    class ENGINE_EXPORT model_file
    {
    public:
      /**
       * \brief Informations about an action.
       * \author Julien Jorge
       */
      struct action
      {
        /** \brief The animation to play. */
        visual::animation* animation;

        /** \brief The sound to play. */
        audio::sound* sound;

      }; // struct action

    private:
      /** \brief The container use for storing our actions. */
      typedef std::map<std::string, action> action_map;

    public:
      model_file( const std::string& file_name );
      ~model_file();

      action& get_action( const std::string& name );

    private:
      void clear();

      void add_action( const std::string& name, visual::animation* animation,
                       const std::string& sound_name );

      void load_model( compiled_file& f );

      unsigned int load_action( compiled_file& f );

      visual::sprite* load_sprite( compiled_file& f ) const;
      visual::animation* load_animation( compiled_file& f ) const;

    private:
      /** \brief The set of actions. */
      action_map m_actions;

    }; // class model_file
  } // namespace engine
} // namespace bear

#endif // __ENGINE_MODEL_FILE_HPP__
