/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file static_text.cpp
 * \brief Implementation of the bear::gui::static_text class.
 * \author Julien Jorge
 */
#include "gui/static_text.hpp"

#include <claw/assert.hpp>
#include "text/text_metric.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owner The component containing this component.
 */
bear::gui::static_text::static_text( visual_component* owner )
  : visual_component(owner), m_auto_size(false), m_auto_expand(false)
{

} // static_text::static_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owner The component containing this component.
 * \param f The font used to draw the text.
 */
bear::gui::static_text::static_text( visual_component* owner, font_type f )
  : visual_component(owner), m_font(f), m_auto_size(false), m_auto_expand(false)
{
  CLAW_PRECOND( f != NULL );
} // static_text::static_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the Font used to draw the text.
 * \param b The new value.
 */
void bear::gui::static_text::set_font( font_type f )
{
  m_font = f;
} // static_text::set_font()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the status of the auto size mode.
 * \param b The new value.
 */
void bear::gui::static_text::set_auto_size( bool b )
{
  m_auto_size = b;

  if (m_auto_size)
    {
      m_auto_expand = false;
      adjust_size_to_text();
    }
} // static_text::set_auto_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the status of the auto expand mode.
 * \param b The new value.
 */
void bear::gui::static_text::set_auto_expand( bool b )
{
  m_auto_expand = b;

  if (m_auto_expand)
    {
      m_auto_size = false;
      adjust_height_to_text();
    }
} // static_text::set_auto_expand()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the text to draw.
 * \param text The text to draw.
 */
void bear::gui::static_text::set_text( const std::string& text )
{
  m_text = text;

  if (m_auto_size)
    adjust_size_to_text();
  else if (m_auto_expand)
    adjust_height_to_text();
} // static_text::set_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the text.
 */
const std::string& bear::gui::static_text::get_text() const
{
  return m_text;
} // static_text::get_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the font used for displaying the text.
 */
bear::gui::static_text::font_type bear::gui::static_text::get_font() const
{
  return m_font;
} // static_text::get_font()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the height of the text component would have if the width is set
 *        to a given value.
 * \param w The hypothetic width.
 */
claw::math::coordinate_2d<unsigned int>
bear::gui::static_text::get_size_with_max_width( unsigned int w ) const
{
  claw::math::coordinate_2d<unsigned int> result;

  if ( m_font == NULL )
    result.set(0, 0);
  else if ( m_auto_size == true )
    {
      text::text_metric tm( m_text, *m_font );
      result.set( tm.width(), tm.height() );
    }
  else if ( (m_auto_expand == false) || (w < m_font->get_size().x) )
    result = get_size();
  else
    result = get_auto_size_with_max_width(w);

  return result;
} // static_text::get_size_with_max_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the length of the longest text that can be displayed in this
 *        component.
 * \param text The text on which we work.
 * \param i The position in \a text
 * \return the position of the first character out of the component.
 */
std::size_t bear::gui::static_text::get_longest_text
( const std::string& text, std::size_t i ) const
{
  if ( m_font == NULL )
    return text.size() - i;
  else
    {
      const std::size_t line_length = width() / m_font->get_size().x;
      const std::size_t lines_count = height() / m_font->get_size().y;
      claw::math::coordinate_2d<unsigned int> cursor(0,0);

      while ( (cursor.y < lines_count) && (i!=text.size()) )
        if ( text[i] == '\n' )
          {
            ++i;
            ++cursor.y;
            cursor.x = 0;
          }
        else
          display_word_dummy( text, cursor, i, line_length );

      return i;
    }
} // static_text::get_longest_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the size of the text component would have if the width is set
 *        to a given value and the auto_expand mode is on.
 * \param w The hypothetic maximum width.
 */
claw::math::coordinate_2d<unsigned int>
bear::gui::static_text::get_auto_size_with_max_width( unsigned int w ) const
{
  CLAW_PRECOND( m_font != NULL );

  const std::size_t line_length = w / m_font->get_size().x;
  std::size_t i=0;
  claw::math::coordinate_2d<unsigned int> cursor(0, 0);
  claw::math::coordinate_2d<unsigned int> result(0, 0);

  while ( i!=m_text.size() )
    if ( m_text[i] == '\n' )
      {
        if ( cursor.x * m_font->get_size().x > result.x )
          result.x = cursor.x * m_font->get_size().x;

        ++i;
        ++cursor.y;
        cursor.x = 0;
      }
    else
      {
        const std::size_t max_w =
          display_word_dummy( m_text, cursor, i, line_length );

        if ( max_w * m_font->get_size().x > result.x )
          result.x = max_w * m_font->get_size().x;
      }

  if ( cursor.x == 0 )
    result.y = cursor.y * m_font->get_size().y;
  else
    {
      if ( cursor.x * m_font->get_size().x > result.x )
        result.x = cursor.x * m_font->get_size().x;

      result.y = (cursor.y + 1) * m_font->get_size().y;
    }

  return result;
} // static_text::get_auto_size_with_max_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Simulate the drawing of a word on the screen.
 * \param text The text on which we work.
 * \param cursor (in/out) The position of the cursor in the component.
 * \param i (in/out) Index of the first character of the word. (out) Index of
 *        the next character to print.
 * \param line_length The length of the line.
 * \return The length of the last line displayed.
 */
std::size_t bear::gui::static_text::display_word_dummy
( const std::string& text, claw::math::coordinate_2d<unsigned int>& cursor,
  std::size_t& i, const std::size_t line_length ) const
{
  std::size_t result(0);

  // find the first word
  std::size_t word = text.find_first_not_of(' ', i);

  if (word == std::string::npos)
    {
      result = cursor.x;
      i = text.size();
    }
  else
    {
      if (text[word] == '\n')
        i = word;
      else
        {
          // the end of the word
          std::size_t space = text.find_first_of(" \n", word);

          if (space == std::string::npos)
            space = text.size();

          const std::size_t word_length = space - i;

          // the word fits on the line
          if ( cursor.x + word_length <= line_length )
            result =
              display_word_dummy(text, cursor, i, word_length, line_length );
          else if ( cursor.x == 0 )     // the word doesn't fit on the full line
            result =
              display_word_dummy(text, cursor, i, line_length, line_length );
          else
            {
              // we will retry at the begining of the line in the next loop
              // we remove the spaces at the begining of the line
              result = cursor.x;
              ++cursor.y;
              cursor.x = 0;
              i = word;
            }
        }
    }

  return result;
} // static_text::display_word_dummy()

/*----------------------------------------------------------------------------*/
/**
 * \brief Simulate the drawing of a word.
 * \param text The text on which we work.
 * \param cursor (in/out) The position of the cursor in the component.
 * \param i (in/out) Index of the first character of the word. (out) Index of
 *        the next character to print.
 * \param n Number of characters to print.
 * \param line_length The length of the line.
 * \return The length of the last line displayed.
 */
std::size_t bear::gui::static_text::display_word_dummy
( const std::string& text, claw::math::coordinate_2d<unsigned int>& cursor,
  std::size_t& i, const std::size_t n, const std::size_t line_length ) const
{
  std::size_t result(0);

  cursor.x += n;
  i += n;

  if ( cursor.x == line_length )
    {
      result = cursor.x;
      cursor.x = 0;
      ++cursor.y;

      if ( i < text.size() )
        {
          i = text.find_first_not_of(' ', i);

          if ( i == std::string::npos )
            i = text.size();
          else if ( text[i] == '\n' )
            ++i;
        }
    }

  return result;
} // static_text::display_word_dummy()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the component on a screen.
 * \param screen The screen on which to draw.
 * \param pos The position of this component in the screen.
 */
void bear::gui::static_text::display
( visual::screen& screen,
  const claw::math::coordinate_2d<unsigned int>& pos ) const
{
  if ( m_font != NULL )
    {
      const unsigned int lines_count = height() / m_font->get_size().y;

      std::size_t i=0;
      claw::math::coordinate_2d<unsigned int> cursor(0,0);

      while ( (cursor.y < lines_count) && (i!=m_text.size()) )
        if ( m_text[i] == '\n' )
          {
            ++i;
            ++cursor.y;
            cursor.x = 0;
          }
        else
          display_word(screen, pos, cursor, i);
    }
} // static_text::display()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw the next word on the screen.
 * \param screen The screen on which to draw.
 * \param pos The position of this component in the screen.
 * \param cursor (in/out) The position of the cursor in the component.
 * \param i (in/out) Index of the first character of the word. (out) Index of
 *        the next character to print.
 */
void bear::gui::static_text::display_word
( visual::screen& screen,
  const claw::math::coordinate_2d<unsigned int>& pos,
  claw::math::coordinate_2d<unsigned int>& cursor, std::size_t& i ) const
{
  CLAW_PRECOND( m_font != NULL );

  const std::size_t line_length = width() / m_font->get_size().x;

  // find the first word
  std::size_t word = m_text.find_first_not_of(' ', i);

  if (word == std::string::npos)
    i = m_text.size();
  else
    {
      if (m_text[word] == '\n')
        i = word;
      else
        {
          // the end of the word
          std::size_t space = m_text.find_first_of(" \n", word);

          if (space == std::string::npos)
            space = m_text.size();

          const std::size_t word_length = space - i;

          // the word fits on the line
          if ( cursor.x + word_length <= line_length )
            display_word(screen, pos, cursor, i, word_length, line_length );
          else if ( cursor.x == 0 )     // the word doesn't fit on the full line
            display_word(screen, pos, cursor, i, line_length, line_length );
          else
            {
              // we will retry at the begining of the line in the next loop
              // we remove the spaces at the begining of the line
              ++cursor.y;
              cursor.x = 0;
              i = word;
            }
        }
    }
} // static_text::display_word()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw a word on the screen.
 * \param screen The screen on which to draw.
 * \param pos The position of this component in the screen.
 * \param cursor (in/out) The position of the cursor in the component.
 * \param i (in/out) Index of the first character of the word. (out) Index of
 *        the next character to print.
 * \param n Number of characters to print.
 * \param line_length The length of the line.
 */
void bear::gui::static_text::display_word
( visual::screen& screen,
  const claw::math::coordinate_2d<unsigned int>& pos,
  claw::math::coordinate_2d<unsigned int>& cursor,
  std::size_t& i, const std::size_t n, const std::size_t line_length ) const
{
  CLAW_PRECOND( m_font != NULL );

  claw::math::coordinate_2d<unsigned int> sprite_pos;

  sprite_pos.x = pos.x + cursor.x * m_font->get_size().x;
  sprite_pos.y = pos.y + cursor.y * m_font->get_size().y;

  m_font->render_text( screen, sprite_pos, m_text.substr(i, n) );

  cursor.x += n;
  i += n;

  if ( cursor.x == line_length )
    {
      cursor.x = 0;
      ++cursor.y;

      if ( i < m_text.size() )
        {
          i = m_text.find_first_not_of(' ', i);

          if ( i == std::string::npos )
            i = m_text.size();
          else if ( m_text[i] == '\n' )
            ++i;
        }
    }
} // static_text::display_word()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the size of the component to fit the text.
 */
void bear::gui::static_text::adjust_size_to_text()
{
  if ( m_font == NULL )
    set_size(0, 0);
  else
    {
      text::text_metric tm( m_text, *m_font );

      claw::math::coordinate_2d<unsigned int> size( tm.width(), tm.height() );

      set_size( size );
    }
} // static_text::adjust_size_to_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the height of the component to fit the text.
 */
void bear::gui::static_text::adjust_height_to_text()
{
  set_size( get_size_with_max_width(width()) );
} // static_text::adjust_height_to_text()
