/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file font.hpp
 * \brief Implementation of the bear::text::font class.
 * \author Sebastien Angibaud
 */
#include "text/font.hpp"

#include <algorithm>
#include <claw/algorithm.hpp>
#include <claw/functional.hpp>
#include <claw/assert.hpp>
#include <claw/exception.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param image_resources The name of the image resource containing the
 *        characters.
 * \param character_size The size of the characters.
 */
bear::text::font::font( const visual::image& image_resource,
                  const claw::math::coordinate_2d<unsigned int>& character_size)
  : m_character_size(character_size),
    m_sprites(26 + 26 + 10, (visual::sprite*)NULL)
{
  const unsigned int characters_per_line =
    image_resource.width() / m_character_size.x;
  unsigned int x = 0;
  unsigned int y = 0;

  for( unsigned int i = 0; i != m_sprites.size(); ++i )
    {
      create_sprite( image_resource, i, x, y );

      ++x;

      if ( x == characters_per_line )
        {
          x = 0;
          ++y;
        }
    }

  create_special_sprites( image_resource, x, y, characters_per_line );
} // font::font()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 * \param that The font to copy from.
 */
bear::text::font::font( const font& that )
  : m_character_size(that.m_character_size), m_sprites(that.m_sprites.size())
{
  std::transform( that.m_sprites.begin(), that.m_sprites.end(),
                  m_sprites.begin(), claw::clone<visual::sprite>() );

  sprite_map::const_iterator it;

  for ( it=that.m_special_sprites.begin();
        it!=that.m_special_sprites.end();
        ++it )
    m_special_sprites[it->first] = new visual::sprite(*it->second);
} // font::font()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::text::font::~font()
{
  typedef claw::unary_compose
    < claw::delete_function<visual::sprite*>,
    claw::pair_second<sprite_map::value_type> > delete_function;

  claw::inplace_for_each( m_sprites.begin(), m_sprites.end(),
                          claw::delete_function<visual::sprite*>() );

  claw::inplace_for_each( m_special_sprites.begin(),
                          m_special_sprites.end(), delete_function() );
} // font::~font()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite of a character.
 * \param character The character to get.
 */
const bear::visual::sprite& bear::text::font::operator[](char character) const
{
  return get_sprite(character);
} // font::operator[]

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the size of the characters.
 */
const claw::math::coordinate_2d<unsigned int>&
bear::text::font::get_size() const
{
  return m_character_size;
} // font::get_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw a word on the screen.
 * \param screen The screen on which to draw.
 * \param pos The position where the text begins.
 * \param str The string to write.
 */
void bear::text::font::render_text
( visual::screen& screen, const claw::math::coordinate_2d<unsigned int>& pos,
  const std::string& str ) const
{
  claw::math::coordinate_2d<unsigned int> cur_pos(pos);
  std::string::const_iterator it;

  for (it=str.begin(); it!=str.end(); ++it)
    {
      screen.render( cur_pos, get_sprite(*it) );
      cur_pos.x += m_character_size.x;
    }
} // font::render_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite of a character.
 * \param character The character to get.
 */
const bear::visual::sprite& bear::text::font::get_sprite( char character ) const
{
  if ( ( character >= 'a' ) && ( character <= 'z' ) )
    {
      unsigned int pos = character - 'a';
      return *m_sprites[ pos ];
    }
  else if ( ( character >= 'A' ) && ( character <= 'Z' ) )
    {
      unsigned int pos = character - 'A';
      return *m_sprites[ pos + 26 ];
    }
  else if ( ( character >= '0' ) && ( character <= '9' ) )
    {
      unsigned int pos = character - '0';
      return *m_sprites[ pos + 2 * 26 ];
    }
  else
    {
      sprite_map::const_iterator it;
      it = m_special_sprites.find(character);

      if ( it == m_special_sprites.end() )
        it = m_special_sprites.find(' ');

      return *it->second;
    }
} // font::get_sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite associated with a basic character.
 * \param image_resource The image in which to get the sprite.
 * \param index The index of the character in this->m_sprites.
 * \param pos_x X-coordinate of the character in the image.
 * \param pos_y Y-coordinate of the character in the image.
 */
void bear::text::font::create_sprite
( const visual::image& image_resource, const unsigned int index,
  const unsigned int pos_x, const unsigned int pos_y )
{
  CLAW_PRECOND( !m_sprites[index] );

  claw::math::rectangle<unsigned int> box( m_character_size.x * pos_x,
                                           m_character_size.y * pos_y,
                                           m_character_size.x,
                                           m_character_size.y );

  m_sprites[index] = new visual::sprite( image_resource, box );
} // font::create_sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite associated with a special character.
 * \param image_resource The image in which to get the sprite.
 * \param c The character value.
 * \param pos_x X-coordinate of the character in the image.
 * \param pos_y Y-coordinate of the character in the image.
 */
void bear::text::font::create_sprite
( const visual::image& image_resource, char c, const unsigned int pos_x,
  const unsigned int pos_y )
{
  CLAW_PRECOND( m_special_sprites.find(c) == m_special_sprites.end() );

  claw::math::rectangle<unsigned int> box( m_character_size.x * pos_x,
                                           m_character_size.y * pos_y,
                                           m_character_size.x,
                                           m_character_size.y );

  m_special_sprites[c] = new visual::sprite( image_resource, box );
} // font::create_sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprites for the special characters.
 * \param image_resource The image in which to get the sprites.
 */
void bear::text::font::create_special_sprites
( const visual::image& image_resource, unsigned int x, unsigned int y,
  unsigned int characters_per_line )
{
  const char* characters = " !\"#$%&\'()*+,-./:;<=>?@[\\]^_`{|}~";

  for (const char* p = characters; *p != '\0'; ++p)
    {
      create_sprite( image_resource, *p, x, y );

      ++x;

      if ( x == characters_per_line )
        {
          x = 0;
          ++y;
        }
    }
} // font::create_special_sprites()
