/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file forced_join.cpp
 * \brief Implementation of the bear::universe::forced_join class.
 * \author Julien Jorge.
 */
#include "universe/forced_join.hpp"

/*---------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param moving_item The controlled item.
 * \param target The item to reach.
 * \steps steps Number of steps, maximum, to reach the target.
 */
bear::universe::forced_join::forced_join
( physical_item& moving_item, const physical_item& target, unsigned int steps )
  : forced_movement( moving_item ), m_target( target ),
    m_remaining_steps( steps ), m_playing(false)
{

} // forced_join::forced_join()

/*---------------------------------------------------------------------------*/
/**
 * \brief Start the movement.
 */
void bear::universe::forced_join::start()
{
  m_playing = true;
} // forced_join::start()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do an iteration of the movement.
 */
void bear::universe::forced_join::next_position()
{
  if (m_remaining_steps && m_playing)
    {
      position_type target_center, item_center;
      bear::universe::position_type dp;

      --m_remaining_steps;

      target_center = m_target.get_center_of_mass();
      item_center = m_moving_item.get_center_of_mass();

      if ( m_remaining_steps )
        dp = (target_center - item_center) / m_remaining_steps;
      else
        dp = (target_center - item_center);

      m_moving_item.set_position( m_moving_item.get_position() + dp );

      if ( (item_center + dp) == target_center )
        m_remaining_steps = 0;
    }
} // forced_join::progress()

/*---------------------------------------------------------------------------*/
/**
 * \brief Stop the movement.
 */
void bear::universe::forced_join::stop()
{
  m_playing = false;
} // forced_join::stop()

