/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file forced_rotation.cpp
 * \brief Implementation of the bear::universe::forced_rotation class.
 * \author Julien Jorge
 */
#include "universe/forced_rotation.hpp"
#include <claw/math.hpp>
#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param moving_item The item to wich to movement is applied.
 * \param center The item to use as the center of the rotation.
 */
bear::universe::forced_rotation::forced_rotation( physical_item& moving_item,
                                            const physical_item& center,
                                            double distance )
  : forced_movement(moving_item), m_center(center), m_start_angle(0),
    m_end_angle(0), m_step_length(0), m_distance(distance), m_loop_back(false),
    m_playing(false)
{
  m_angle = m_start_angle;

  m_delta_center =
    m_moving_item.get_center_of_mass() - m_moving_item.get_position();
} // forced_rotation::forced_rotation() [constructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the caracteristics of the rotation.
 * \param start_angle Initial angle.
 * \param end_angle Final angle.
 * \param steps Number of steps in the path.
 */
void bear::universe::forced_rotation::set_path
( double start_angle, double end_angle, unsigned int steps )
{
  m_start_angle = start_angle;
  m_end_angle = end_angle;

  if (!steps)
    m_step_length = 0;
  else
    m_step_length = (m_end_angle - m_start_angle) / (float)steps;
} // forced_rotation::set_start_angle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set loop_back mode.
 * \brief b True if we must play the movement backward when the en is reached.
 */
void bear::universe::forced_rotation::set_loop_back( bool b )
{
  m_loop_back = b;
} // forced_rotation::set_loop_back()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start the movement.
 */
void bear::universe::forced_rotation::start()
{
  m_playing = true;
  set_item_position();
} // forced_rotation::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do an iteration of the movement.
 */
void bear::universe::forced_rotation::next_position()
{
  if (m_playing)
    {
      update_angle();
      set_item_position();
    }
} // forced_rotation::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Stop the movement.
 */
void bear::universe::forced_rotation::stop()
{
  m_playing = false;
} // forced_rotation::stop()

/*----------------------------------------------------------------------------*/
/**
 * \brief Calculate the next angle.
 */
void bear::universe::forced_rotation::update_angle()
{
  m_angle += m_step_length;

  if ( ((m_angle > m_end_angle) && (m_end_angle > m_start_angle))
       || ((m_angle < m_end_angle) && (m_end_angle < m_start_angle)) )
    {
      if (m_loop_back)
        {
          m_angle = m_end_angle;
          m_step_length = -m_step_length;
        }
      else
        m_angle = m_start_angle;
    }
  else if ( ((m_angle < m_start_angle) && (m_start_angle < m_end_angle))
            || ((m_angle > m_start_angle) && (m_start_angle > m_end_angle)) )
    {
      m_angle = m_start_angle;
      m_step_length = -m_step_length;
    }
} // forced_rotation::update_angle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the moving item.
 */
void bear::universe::forced_rotation::set_item_position()
{
  bear::universe::position_type pos;
  pos = m_center.get_center_of_mass();

  pos.x += (m_distance * cos( m_angle ));
  pos.y += (m_distance * sin( m_angle ));

  pos -= m_delta_center;

  m_moving_item.set_center_of_mass(pos);
} // forced_rotation::set_item_position()
