/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file collision_detection.hpp
 * \brief This class detects collisions between items from a given set.
 * \author Julien Jorge
 */
#ifndef __UNIVERSE_COLLISION_DETECTION_HPP__
#define __UNIVERSE_COLLISION_DETECTION_HPP__

#include "universe/physical_item.hpp"

#include <claw/avl.hpp>
#include <list>

#include "universe/class_export.hpp"

namespace bear
{
  namespace universe
  {
    /**
     * \brief This class detects collisions between items from a given set.
     * \author Julien Jorge
     */
    class UNIVERSE_EXPORT collision_detection
    {
    private:
      /** \brief The type of an item in the m_{static,mobile}_items lists. We
          store its state too when the item is added. */
      class UNIVERSE_EXPORT item_set_item
        : public std::pair<physical_item*, physical_item_state>
      {
      public:
        /** \brief The type of the parent class. */
        typedef std::pair<physical_item*, physical_item_state> super;

      public:
        item_set_item( first_type f, second_type s );

        bool operator<( const item_set_item& that ) const;

      }; // class item_set_item;

      /** \brief The type of the m_{static,mobile}_items sets. */
      typedef claw::avl<item_set_item> item_set;

      /** \brief The type of the lists passed to add_items. */
      typedef std::list<physical_item*> item_list;

      typedef std::pair<physical_item*, physical_item*> item_pair;

    public:
      void add_items( const item_list& static_items );

      void detect_collisions();

    private:
      void clear_lists();

      bool detect_collision_remaining( const item_set::const_iterator& item, 
                                       item_set& items);
      bool detect_collision_all( const item_set_item& item, item_set& items  );

      void final_collision_detection();

      bool detect_collision( const item_set_item& item_1,
                             const item_set_item& item_2 );
      void apply_collision( const item_set_item& item_1,
                            const item_set_item& item_2 ) const;

    private:
      /** \brief The solid items without mass : 
          can't collide with another from this
          list. */
      item_set m_fixed_move_items;

      /** \brief The immaterial items without mass : 
          can't collide with another from this
          list. */
      item_set m_fixed_not_move_items;

      /** \brief The solid items with speed : can collide with anything. */
      item_set m_mobile_move_items;

      /** \brief The immaterial items with speed : can collide with anything. */
      item_set m_mobile_not_move_items;

      claw::avl<item_pair> m_corrected;

    }; // class collision_detection
  } // namespace universe
} // namespace bear

#endif // __UNIVERSE_COLLISION_DETECTION_HPP__
