/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file align_take_speed.hpp
 * \brief This collision event align the other item and take a percentage of his
 *        acceleration.
 * \author Julien Jorge
 */
#ifndef __UNIVERSE_ALIGN_TAKE_SPEED_HPP__
#define __UNIVERSE_ALIGN_TAKE_SPEED_HPP__

#include "universe/collision_event/collision_event_align.hpp"

namespace bear
{
  namespace universe
  {
    /**
     * \brief This collision event align the other item and take a percentage of
     *        his speed.
     *
     * The speed is transfered from one item to the other. For example, if you
     * set a ratio of 0.25, the current item will receive a speed equal to 25 %
     * of the speed of the other item, and the other item will have its speed
     * set to 75 %.
     *
     * Ratios greater that one or less than zero are allowed. The other item's
     * speed is, in all case, multiplied by (1 - ratio).
     *
     * \author Julien Jorge
     */
    template<class Alignment>
    class align_take_speed :
      public collision_event_align<Alignment>
    {
    private:
      /** \brief The type of the parent class. */
      typedef collision_event_align<Alignment> super;

    public:
      align_take_speed( double ratio );

      void execute( const collision_info& info, physical_item& self,
                    physical_item& that ) const;

    private:
      /** \brief The coefficient to apply to the other item's speed before
          taking it for us. */
      const double m_ratio;

    }; // class align_take_speed

    /**
     * \brief A class to help generating the align_take_speed<Alignment>
     * instance, where \a Alignment is generated from a universe::zone value.
     */
    class align_take_speed_type
    {
    public:
      template<zone::position Z>
      class type :
        public align_take_speed< typename alignment_type<Z>::type >
      {
      private:
        /** \brief The type of the parent class. */
        typedef align_take_speed< typename alignment_type<Z>::type > super;

      public:
        /**
         * \brief Constructor.
         * \remark Just pass the parameters to the parent class.
         */
        type( double ratio = 2 )
          : super( ratio )
        { }
      }; // class type
    }; // class align_take_speed_type

    typedef
    align_event_generator<align_take_speed_type>
    event_generator_align_take_speed;

  } // namespace universe
} // namespace bear

#include "universe/collision_event/code/align_take_speed.tpp"

#endif // __UNIVERSE_ALIGN_TAKE_SPEED_HPP__
