/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file forced_rotation.hpp
 * \brief A forced movement that will make the current item rotating around
 *        another item.
 * \author Julien Jorge
 */
#ifndef __UNIVERSE_FORCED_ROTATION_HPP__
#define __UNIVERSE_FORCED_ROTATION_HPP__

#include "universe/forced_movement.hpp"

#include "universe/class_export.hpp"

namespace bear
{
  namespace universe
  {
    /**
     * \brief A forced movement that will make the current item rotating around
     *        another item.
     *
     * The positions considered for the items are their center of mass.
     *
     * \author Julien Jorge
     */
    class UNIVERSE_EXPORT forced_rotation:
      public forced_movement
    {
    public:
      forced_rotation( physical_item& moving_item, const physical_item& center,
                       double distance );

      void set_path( double start_angle, double end_angle, unsigned int steps );
      void set_loop_back( bool b );

      virtual void start();
      virtual void next_position();
      virtual void stop();

    protected:
      void update_angle();
      void set_item_position();

    private:
      /** \brief The item we are turning around. */
      const physical_item& m_center;

      /** \brief Initial angle. */
      double m_start_angle;

      /** \brief End angle. */
      double m_end_angle;

      /** \brief Increment for the current angle, to progress. */
      double m_step_length;

      /** \brief Current angle. */
      double m_angle;

      /** \brief Distance of the moving item from the center of the rotation. */
      const double m_distance;

      /** \brief Must we play the rotation backward when the end is readched or
          just come back to the start angle ? */
      bool m_loop_back;

      /** \brief True if the movement is currently playing. */
      bool m_playing;

      /** \brief Distance from the top-left corner of the moving item and its
          center of mass. */
      universe::position_type m_delta_center;

    }; // class forced_rotation
  } // namespace universe
} // namespace bear

#endif // __UNIVERSE_FORCED_ROTATION_HPP__
