/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file physical_item.hpp
 * \brief The common interface for all the physical items living in our
 *        universe.
 * \author Julien Jorge
 */
#ifndef __UNIVERSE_PHYSICAL_ITEM_HPP__
#define __UNIVERSE_PHYSICAL_ITEM_HPP__

#include <claw/math.hpp>
#include <claw/smart_ptr.hpp>

#include <vector>
#include <list>
#include <set>

#include "universe/physical_item_state.hpp"
#include "universe/zone.hpp"

namespace bear
{
  namespace universe
  {
    class forced_movement;
    class collision_event;
    class base_link;

    /**
     * \brief The common interface for all the physical items living in our
     *        universe.
     * \author Julien Jorge
     */
    class UNIVERSE_EXPORT physical_item :
      public physical_item_state
    {
    public:
      /** \brief The type of the class that stores our fields. */
      typedef physical_item_state state_type;

      /** \brief The type of the list in which we store the links. */
      typedef std::list<base_link*> link_list_type;

      /** \brief Const iterator on the list of links. */
      typedef link_list_type::const_iterator const_link_iterator;

      typedef claw::memory::smart_ptr<collision_event> collision_event_ptr;

    public:
      physical_item();
      physical_item( const physical_item& that );
      virtual ~physical_item();

      forced_movement& get_forced_movement();
      void set_forced_movement( forced_movement& m );
      bool have_forced_movement() const;
      void clear_forced_movement();

      virtual void move( time_type elapsed_time );
      virtual void collision( physical_item& that, const state_type& old_self,
                              const state_type& old_that );

      // public only for base_link
      void add_link( base_link& link );
      void remove_link( base_link& link );
      // -end- public only for base_link

      const_link_iterator links_begin() const;
      const_link_iterator links_end() const;

      void add_dependent_item( physical_item_state& item);
      void remove_dependent_item( physical_item_state& item);
      virtual
      void get_dependent_items
      ( std::set<physical_item_state*>& items_set ) const;

    protected:
      void default_move( time_type elapsed_time );
      void set_collision_event( zone::position z, collision_event* event );

      void default_collision_event
      ( physical_item& that, const state_type& old_self,
        const state_type& old_that );

    private:
      void remove_all_links();

      void update_position( time_type elapsed_time );
      void adjust_cinetic();

    private:
      /** \brief Forced movement applied to this item, NULL if none. */
      forced_movement* m_forced_movement;

      /** \brief Default events for collisions. */
      std::vector<collision_event_ptr> m_collision_event;

      /** \brief The links concerning the item. */
      link_list_type m_links;

      /** \brief List of dependent items. */
      std::set<physical_item_state*> m_dependent_items;

      /** \brief Value under which the acceleration is considered as zero. */
      static const force_type::value_type c_acceleration_epsilon;

      /** \brief Value under which the speed is considered as zero. */
      static const speed_type::value_type c_speed_epsilon;

    }; // class physical_item
  } // namespace universe
} // namespace bear

#endif // __UNIVERSE_PHYSICAL_ITEM_HPP__
