/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file image_manager.cpp
 * \brief Implementation of the image_manager class.
 * \author Julien Jorge
 */

#include "visual/image_manager.hpp"
#include <claw/targa.hpp>
#include <claw/assert.hpp>
#include <claw/exception.hpp>
#include <fstream>

/*---------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::visual::image_manager::~image_manager()
{
  clear();
} // image_manager::~image_manager()

/*---------------------------------------------------------------------------*/
/**
 * \brief Deletes all images.
 * \post m_images.empty == true.
 */
void bear::visual::image_manager::clear()
{
  std::map<std::string, image*>::iterator it;

  for (it=m_images.begin(); it!=m_images.end(); ++it)
    delete it->second;

  m_images.clear();
} // image_manager::clear()

/*---------------------------------------------------------------------------*/
/**
 * \brief Add an image to the cache.
 * \param name The name of the loaded image.
 * \param file A stream containing the file to load.
 * \pre name is not used by another image.
 * \post get_image(name) is the image in file_name.
 */
void bear::visual::image_manager::load_image( const std::string& name,
                                        std::istream& file )
{
  CLAW_PRECOND( !exists(name) );

  claw::graphic::image img(file);
  m_images[name] = new image(img);
} // image_manager::load_image()

/*---------------------------------------------------------------------------*/
/**
 * \brief Get an image from the cache.
 * \param name The name of the image to get.
 * \pre There is an image called "name".
 */
const bear::visual::image& bear::visual::image_manager::get_image
(const std::string& name) const
{
  CLAW_PRECOND( exists(name) );

  return *( m_images.find(name)->second );
} // image_manager::get_image()

/*---------------------------------------------------------------------------*/
/**
 * \brief Tell if an image exists.
 * \param name The name of the image to find.
 */
bool bear::visual::image_manager::exists( const std::string& name ) const
{
  return m_images.find(name) != m_images.end();
} // image_manager::exists()
