/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file sprite.cpp
 * \brief Implementation of the  sprite class.
 * \author Julien Jorge
 */
#include <claw/assert.hpp>
#include "visual/sprite.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param img The image for the sprite.
 * \param clip_rectangle Area of "image" for the sprite.
 * \pre clip_rectangle is valid (in the image).
 */
bear::visual::sprite::sprite
( const image& img, const claw::math::rectangle<unsigned int>& clip_rectangle )
  : m_image(img), m_clip_rectangle(clip_rectangle),
    m_size(m_clip_rectangle.width, m_clip_rectangle.height), m_flip(false),
    m_mirror(false), m_alpha_blend(1.0)
{
  CLAW_PRECOND( m_clip_rectangle.position.x
                + m_clip_rectangle.width <= m_image.width() );
  CLAW_PRECOND( m_clip_rectangle.position.y
                + m_clip_rectangle.height <= m_image.height() );
} // sprite::sprite() [constructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param img The image for the sprite.
 */
bear::visual::sprite::sprite( const image& img )
  : m_image(img), m_clip_rectangle(0, 0, img.width(), img.height()),
    m_size(m_clip_rectangle.width, m_clip_rectangle.height), m_flip(false),
    m_mirror(false), m_alpha_blend(1.0)
{

} // sprite::sprite() [constructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param spr The sprite.
 */
bear::visual::sprite::sprite
( const sprite& spr )
  : m_image(spr.get_image()), m_clip_rectangle(spr.clip_rectangle()),
    m_size(spr.get_size()), m_flip(spr.is_flipped()),
    m_mirror(spr.is_mirrored()), m_alpha_blend(spr.get_alpha_blend())
{  
} // sprite::sprite() [constructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Get sprite's translucy.
 */
double bear::visual::sprite::get_alpha_blend() const
{
  return m_alpha_blend;
} // sprite::get_alpha_blend()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set sprite's translucy.
 */
void bear::visual::sprite::set_alpha_blend( double alpha_blend )
{
  m_alpha_blend = alpha_blend;
} // sprite::set_alpha_blend()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the rectangular region of the picture used by this sprite.
 */
const claw::math::rectangle<unsigned int>&
bear::visual::sprite::clip_rectangle() const
{
  return m_clip_rectangle;
} // sprite::clip_rectangle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the picture used by this sprite.
 */
const bear::visual::image& bear::visual::sprite::get_image() const
{
  return m_image;
} // sprite::get_image()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the size of this sprite.
 */
const claw::math::coordinate_2d<unsigned int>&
bear::visual::sprite::get_size() const
{
  return m_size;
} // sprite::get_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the size of the sprite.
 * \param s The new size of the sprite.
 */
void bear::visual::sprite::set_size
( const claw::math::coordinate_2d<unsigned int>& s )
{
  m_size = s;
} // sprite::set_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the size of the sprite.
 * \param w The new width of the sprite.
 * \param h The new height of the sprite.
 */
void bear::visual::sprite::set_size( unsigned int w, unsigned int h )
{
  m_size.set(w, h);
} // sprite::set_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the width of the sprite.
 * \param w The new width of the sprite.
 */
void bear::visual::sprite::set_width( unsigned int w )
{
  m_size.x = w;
} // sprite::set_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the height of the sprite.
 * \param h The new height of the sprite.
 */
void bear::visual::sprite::set_height( unsigned int h )
{
  m_size.y = h;
} // sprite::set_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the width of this sprite.
 */
unsigned int bear::visual::sprite::width() const
{
  return m_size.x;
} // sprite::width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the height of this sprite.
 */
unsigned int bear::visual::sprite::height() const
{
  return m_size.y;
} // sprite::height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set vertical inversion.
 * \param b The new value for the inversion.
 */
void bear::visual::sprite::flip( bool b )
{
  m_flip = b;
} // sprite::flip()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set horizontal inversion.
 * \param b The new value for the inversion.
 */
void bear::visual::sprite::mirror( bool b )
{
  m_mirror = b;
} // sprite::mirror()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get vertical inversion.
 */
bool bear::visual::sprite::is_flipped() const
{
  return m_flip;
} // sprite::is_flipped()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get horizontal inversion.
 */
bool bear::visual::sprite::is_mirrored() const
{
  return m_mirror;
} // sprite::is_mirrored()
