/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file screen.hpp
 * \brief The screen surface, where drawing & blitting is done.
 * \author Julien Jorge
 */
#ifndef __VISUAL_SCREEN_HPP__
#define __VISUAL_SCREEN_HPP__

#include "visual/sprite.hpp"
#include "visual/screen_effect.hpp"
#include <claw/math.hpp>

#include "visual/class_export.hpp"

namespace bear
{
  namespace visual
  {
    /**
     * \brief The screen surface, where drawing & blitting is done.
     * \author Julien Jorge
     */
    class screen
    {
    private: // Types
      /** \brief Defined the current screen process. */
      enum screen_status
        {
          SCREEN_IDLE,
          SCREEN_RENDER,
          SCREEN_EFFECTS
        }; // enum screen_status

    public:
      VISUAL_EXPORT static void initialize();
      VISUAL_EXPORT static void release();

      VISUAL_EXPORT screen( const claw::math::coordinate_2d<unsigned int>& size,
                            const std::string& title="", bool full=false );

      VISUAL_EXPORT void resize_view(unsigned int width, unsigned int height);

      VISUAL_EXPORT void fullscreen( bool b );

      VISUAL_EXPORT
      const claw::math::coordinate_2d<unsigned int>& get_size() const;

      VISUAL_EXPORT void begin_render();
      VISUAL_EXPORT void render( const claw::math::coordinate_2d<int>& pos,
                                 const sprite& s, double a = 0 );
      VISUAL_EXPORT void end_render();

      VISUAL_EXPORT void begin_effects();
      VISUAL_EXPORT void apply_effect( screen_effect& effect );
      VISUAL_EXPORT void end_effects();

      template<typename Iterator>
      void draw_line( const claw::graphic::pixel32& color,
                      Iterator first, Iterator last, double w = 1.0 );
      VISUAL_EXPORT void shot( const std::string& bitmap_name ) const;
      VISUAL_EXPORT void shot( claw::graphic::image& img ) const;

    private:
      VISUAL_EXPORT void render_sprite
      ( const claw::math::coordinate_2d<int>& pos, const sprite& s, double a );

      VISUAL_EXPORT void render_image
      ( const claw::math::coordinate_2d<GLdouble> render_coord[],
        const claw::math::box_2d<unsigned int>& clip,
        const claw::math::coordinate_2d<unsigned int>& tex_size);

      VISUAL_EXPORT claw::math::coordinate_2d<GLdouble>
      rotate( const claw::math::coordinate_2d<GLdouble>& pos, GLdouble a,
              const claw::math::coordinate_2d<GLdouble>& center ) const;

      VISUAL_EXPORT void failure_check( const std::string& where ) const;

    private: // Attributes
      /** \brief True if we are rendering. */
      screen_status m_mode;

      /** \brief The width and height of the screen. */
      const claw::math::coordinate_2d<unsigned int> m_size;

    }; // class screen
  } // namespace visual
} // namespace bear

#include "visual/code/screen.tpp"

#endif // __VISUAL_SCREEN_HPP__
