/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file slope.cpp
 * \brief Implementation of the bear::slope class.
 * \author Julien Jorge
 */
#include "generic_items/invisible_slope.hpp"

#include "engine/export.hpp"

#include "universe/collision_event/horizontal_nearest_align_stop.hpp"

BASE_ITEM_EXPORT( invisible_slope, bear )

/*----------------------------------------------------------------------------*/
const bear::universe::coordinate_type bear::invisible_slope::s_line_width = 10;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::invisible_slope::invisible_slope()
  : m_steepness(0), m_opposite_side_is_solid(false),
    m_left_side_is_solid(false), m_right_side_is_solid(false)
{

} // invisible_slope::invisible_slope()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c <real>.
 * \param name The name of the field to set.
 * \param value The new value of the field.
 */
bool
bear::invisible_slope::set_real_field( const std::string& name, double value )
{
  bool result = true;

  if ( name == "steepness" )
    m_steepness = value;
  else
    result = super::set_real_field(name, value);

  return result;
} // invisible_slope::set_real_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c bool.
 * \param name The name of the field to set.
 * \param value The new value of the field.
 */
bool
bear::invisible_slope::set_bool_field( const std::string& name, bool value )
{
  bool result = true;

  if ( name == "is_ground" )
    m_is_ground = value;
  else if ( name == "opposite_side_is_solid" )
    m_opposite_side_is_solid = value;
  else if ( name == "left_side_is_solid" )
    m_left_side_is_solid = value;
  else if ( name == "right_side_is_solid" )
    m_right_side_is_solid = value;
  else
    result = super::set_u_integer_field(name, value);

  return result;
} // invisible_slope::set_bool_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is correctly initialized.
 */
bool bear::invisible_slope::is_valid() const
{
  return (m_steepness != 0) && super::is_valid();
} // invisible_slope::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the item.
 */
void bear::invisible_slope::start()
{
  super::start();

  if (m_is_ground)
    {
      if ( m_steepness > 0 )
        set_ground_down();
      else
        set_ground_up();
    }
  else
    {
      if ( m_steepness > 0 )
        set_ceiling_up();
      else
        set_ceiling_down();
    }
} // invisible_slope::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for a ground that goes down on the right.
 */
void bear::invisible_slope::set_ground_down()
{
  const collision_event_slope::line_type::value_type w = get_width();
  const collision_event_slope::line_type::value_type h = m_steepness;

  const collision_event_slope::line_type line( 0, 0, w, h );
  m_line = line;

  set_collision_event( universe::zone::top_zone,
                       new collision_event_slope( line, s_line_width ) );

  // bottom edge
  if (m_opposite_side_is_solid)
    set_collision_event
      ( universe::zone::bottom_zone, new ce_align_stop_bottom );

  // left and right zones
  if (m_left_side_is_solid)
    set_collision_event
      ( universe::zone::middle_left_zone, new ce_align_stop_left );
  /*else
    set_collision_event
      ( universe::zone::middle_left_zone, new ce_align_stop_top );*/

  if (m_right_side_is_solid)
    set_collision_event
      ( universe::zone::middle_right_zone,
        new ce_condition_bottom_lower
        ( new universe::bottom_contact_is_lower(m_steepness + s_line_width),
          new collision_event_slope( line, s_line_width ),
          new ce_align_stop_right )  );
  else
    set_collision_event( universe::zone::middle_right_zone,
                         new collision_event_slope( line, s_line_width ) );

  // middle zone
  create_middle_ground_collision_event( line );
} // invisible_slope::set_ground_down()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for a ground that goes up on the right.
 */
void bear::invisible_slope::set_ground_up()
{
  const collision_event_slope::line_type::value_type y = -m_steepness;
  const collision_event_slope::line_type::value_type w = get_width();
  const collision_event_slope::line_type::value_type h = m_steepness;

  const collision_event_slope::line_type line( 0, y, w, h );
  m_line = line;

  set_collision_event( universe::zone::top_zone,
                       new collision_event_slope( line, s_line_width ) );

  // bottom edge
  if (m_opposite_side_is_solid)
    set_collision_event
      ( universe::zone::bottom_zone, new ce_align_stop_bottom );

  // left and right zones
  if (m_right_side_is_solid)
    set_collision_event
      ( universe::zone::middle_right_zone, new ce_align_stop_right );

  if (m_left_side_is_solid)
    set_collision_event
      ( universe::zone::middle_left_zone,
        new ce_condition_bottom_lower
        ( new universe::bottom_contact_is_lower(y + s_line_width),
          new collision_event_slope( line, s_line_width ),
          new ce_align_stop_left )  );
  else
    set_collision_event( universe::zone::middle_left_zone,
                         new collision_event_slope( line, s_line_width ) );

  // middle zone
  create_middle_ground_collision_event( line );
} // invisible_slope::set_ground_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for a ceiling that goes down on the right.
 */
void bear::invisible_slope::set_ceiling_down()
{
  const collision_event_slope::line_type::value_type x = get_width();
  const collision_event_slope::line_type::value_type w = -get_width();

  // steepness is reversed for ceilings
  const collision_event_slope::line_type::value_type h = -m_steepness;

  const collision_event_slope::line_type line( x, 0, w, h );
  m_line = line;

  set_collision_event( universe::zone::bottom_zone,
                       new collision_event_slope( line, s_line_width ) );
  set_collision_event( universe::zone::middle_left_zone,
                       new collision_event_slope( line, s_line_width ) );
  set_collision_event( universe::zone::middle_zone,
                       new collision_event_slope( line, s_line_width ) );
} // invisible_slope::set_ceiling_down()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the collision events for a ceiling that goes up on the right.
 */
void bear::invisible_slope::set_ceiling_up()
{
  const collision_event_slope::line_type::value_type x = get_width();
  const collision_event_slope::line_type::value_type y = get_height();
  const collision_event_slope::line_type::value_type w = -get_width();

  // steepness is reversed for ceilings
  const collision_event_slope::line_type::value_type h = -m_steepness;

  const collision_event_slope::line_type line( x, y, w, h );
  m_line = line;

  set_collision_event( universe::zone::bottom_zone,
                       new collision_event_slope( line, s_line_width ) );
  set_collision_event( universe::zone::middle_right_zone,
                       new collision_event_slope( line, s_line_width ) );
  set_collision_event( universe::zone::middle_zone,
                       new collision_event_slope( line, s_line_width ) );
} // invisible_slope::set_ceiling_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the collision event for the middle zone in the case of a
 *        ground.
 * \param line The line representing the invisible_slope.
 *
 * The collision event of the middle zone is dependant on the position of the
 * colliding item. If the bottom of the item is above the line, 
 * the invisible_slope alignement is applied. Otherwise (the item is below the
 * invisible_slope), the colliding item is aligned to the nearer solid edge of
 * the 'invisible_slope' item.
 */
void bear::invisible_slope::create_middle_ground_collision_event
( const collision_event_slope::line_type& line )
{
  if ( m_left_side_is_solid || m_right_side_is_solid )
    {
      collision_event_slope::line_type bound(line);
      bound.origin.y += s_line_width;

      universe::bottom_contact_is_lower* condition =
        new universe::bottom_contact_is_lower(bound);

      universe::collision_event* then_part =
        new collision_event_slope( line, s_line_width );

      universe::collision_event* else_part =
        new universe::horizontal_nearest_align_stop
        ( m_left_side_is_solid, m_right_side_is_solid );

      set_collision_event
        ( universe::zone::middle_zone, new ce_condition_bottom_lower
          ( condition, then_part, else_part ) );
    }
  else
    set_collision_event( universe::zone::middle_zone,
                         new collision_event_slope( line, s_line_width ) );
} // invisible_slope::create_middle_ground_collision_event()
