/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file level_loader_item.cpp
 * \brief Implementation of the bear::level_loader_item class.
 * \author Julien Jorge
 */
#include "generic_items/level_loader_item.hpp"

#include "engine/resource_pool.hpp"
#include "engine/game.hpp"

#include "engine/export.hpp"

BASE_ITEM_EXPORT( level_loader_item, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::level_loader_item::level_loader_item()
  : m_level_loader(NULL), m_level_file(NULL), m_level_stream(NULL),
    m_level(NULL), m_time(0), m_delay(0)
{
  set_mass(0);
} // level_loader_item::level_loader_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::level_loader_item::~level_loader_item()
{
  if ( m_level_loader != NULL )
    delete m_level_loader;

  if ( m_level_file != NULL )
    delete m_level_file;

  if ( m_level_stream != NULL )
    delete m_level_stream;

  if ( m_level != NULL )
    delete m_level;
} // level_loader_item::~level_loader_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialise the item.
 */
void bear::level_loader_item::start()
{
  m_level_stream = new std::stringstream();

  engine::resource_pool::get_instance().get_file
    ( engine::game::get_instance().get_level_file(m_level_name),
      *m_level_stream );

  m_level_file = new engine::compiled_file( *m_level_stream, true );
  m_level_loader = new engine::level_loader(*m_level_file);
} // level_loader_item::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one step in the progression of the item.
 * \param elapsed_time The elapsed time since the last call.
 */
void bear::level_loader_item::progress( bear::universe::time_type elapsed_time )
{
  m_time += elapsed_time;

  if ( m_level_loader != NULL )
    progress_loading();
  else if ( (m_time * 1000) >= m_delay )
    {
      engine::game::get_instance().set_waiting_level( m_level );
      m_level = NULL;
    }
} // level_loader_item::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type string.
 * \param name The name of the field.
 * \param value The value of the field.
 */
bool bear::level_loader_item::set_string_field
( const std::string& name, const std::string& value )
{
  bool result = true;

  if ( name == "level" )
    m_level_name = value;
  else
    result = super::set_string_field( name, value );

  return result;
} // level_loader_item::level_loader_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <unsigned integer>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::level_loader_item::set_u_integer_field
( const std::string& name, unsigned int value )
{
  bool result;

  if ( name == "delay" )
    {
      m_delay = value;
      result = true;
    }
  else
    result = super::set_u_integer_field( name, value );

  return result;
} // level_loader_item::set_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is well initialised.
 */
bool bear::level_loader_item::is_valid()
{
  return engine::game::get_instance().level_exists(m_level_name)
    && super::is_valid();
} // level_loader_item::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one step in the progression of the level loading.
 */
void bear::level_loader_item::progress_loading()
{
  CLAW_PRECOND( m_level_loader != NULL );
  CLAW_PRECOND( m_level_file != NULL );
  CLAW_PRECOND( m_level_stream != NULL );
  CLAW_PRECOND( m_level == NULL );

  bool stop = false;
  time_ref::time_reference start_time, current_time;

  start_time.set();
  const unsigned int max_t = engine::game::get_instance().get_time_step() / 2;

  do
    {
      stop = m_level_loader->one_step();
      current_time.set();
    }
  while ( !stop && (current_time - start_time < max_t) );

  if (stop)
    {
      m_level = new engine::level( *m_level_loader, m_level_name );

      delete m_level_loader;
      m_level_loader = NULL;

      delete m_level_file;
      m_level_file = NULL;

      delete m_level_stream;
      m_level_stream = NULL;
    }
} // level_loader_item::progress_loading()

