/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file timer_item_level.cpp
 * \brief Implementation of the bear::timer_item class.
 * \author Sbastien Angibaud
 */
#include "generic_items/timer_item.hpp"
#include "engine/export.hpp"

BASE_ITEM_EXPORT( timer_item, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Apply the message to a timer_item.
 * \param that The timer_item to which we apply the message.
 */
bool
bear::timer_item::msg_get_value::apply_to( communication::messageable& that )
{
  CLAW_PRECOND( dynamic_cast<timer_item*>(&that) != NULL );

  timer_item* item = (timer_item*)&that;
  value = item->get_time();

  return true;
} // timer_item::msg_get_value::apply_to()




/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::timer_item::timer_item()
  : m_time(0), m_save_time(0), m_countdown(false)
{
  set_phantom(true);
  set_can_move_items(false);
  set_global(true);
} // timer_item::timer_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialise the item.
 */
void bear::timer_item::start()
{
  super::start();

  m_first_player = engine::pointer_to_player( engine::player::player_name(1) );
  m_second_player = engine::pointer_to_player( engine::player::player_name(2) );
} // timer_item::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 */
void bear::timer_item::progress( universe::time_type elapsed_time )
{
  if ( m_countdown )
    {
      m_time -= elapsed_time;
      if ( m_time <= 0 )
        {
          m_time = m_save_time;
          if ( m_first_player ) 
            m_first_player->do_action(engine::player::action_die);
          if ( m_second_player ) 
            m_second_player->do_action(engine::player::action_die);
        }
    }
  else
    m_time += elapsed_time;
} // timer_item::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <bool>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::timer_item::set_bool_field
( const std::string& name,bool value )
{
  bool result = true;

  if ( name == "countdown" )
    m_countdown = value;
  else
    result = super::set_bool_field( name, value );
  
  return result;
} // timer_item::set_bool_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type unsigned int.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::timer_item::set_u_integer_field
( const std::string& name, unsigned int value )
{
  bool ok = true;

  if (name == "time")
    {
      m_time = (universe::time_type)value;
      m_save_time = m_time;
    }
  else
    ok = super::set_u_integer_field( name, value );

  return ok;
} // timer_item::set_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is well initialised.
 */
bool bear::timer_item::is_valid() const
{
  return !get_name().empty() && super::is_valid();
} // timer_item::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the time.
 */
bear::universe::time_type bear::timer_item::get_time() const
{
  return m_time;
} // timer_item::get_time()
