/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file collision_event_slope.cpp
 * \brief Implementation of the bear::collision_event_slope class.
 * \author Julien Jorge
 */
#include "generic_items/collision_event/collision_event_slope.hpp"
#include <iostream>
/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param surface The line describing the surface of the slope.
 * \param line_width The width of the line: defines the distance above/below the
 *        line in which the collision is checked when the collision occurs.
 *
 * Grounds have surfaces described from left to right
 * (ie. surface.direction.x > 0). Otherwise the item is considered as a ceiling.
 *
 * The origin of the line is relative to the top_left corner of the \a self item
 * of the execute method.
 */
bear::collision_event_slope::collision_event_slope
( const line_type& surface, universe::coordinate_type line_width )
  : m_surface(surface), m_surface_width(line_width)
{

} // collision_event::collision_event_slope()

/*----------------------------------------------------------------------------*/
/**
 * \brief Align the other item on the slope.
 * \param info Informations on the collision.
 * \param self The first item in the collision.
 * \param that The other item in the collision.
 */
void bear::collision_event_slope::execute
( const universe::collision_info& info, universe::physical_item& self,
  universe::physical_item& that ) const
{
  if ( !that.is_phantom()
       && (that.get_center_of_mass().x >= self.get_position().x)
       && (that.get_center_of_mass().x <= self.get_right()) )
    {
      const claw::math::line_2d<universe::coordinate_type> line
        ( self.get_position() + m_surface.origin, m_surface.direction );
      bool aligned = true;

      if ( m_surface.direction.x > 0 )
        {
          if ( item_crossed_up_down(info, self, that, line) )
            {
              if ( ( m_surface.direction.x * 1.7 <= m_surface.direction.y ) || 
                   ( - m_surface.direction.x * 1.7 <= m_surface.direction.y ) )
                align_on_top(that, self, line);
              else
                align_replace(info, that, self, line);
            }
          else
            aligned = false;
        }
      else if (m_surface.direction.x < 0 )
        {
          if ( item_crossed_down_up(info, self, that, line) )
            align_below(that, self, line);
          else
            aligned = false;
        }
      else
        aligned = false;

      if (aligned)
        {
          universe::speed_type speed( that.get_speed() );
          universe::force_type accel( that.get_acceleration() );

          speed.y = accel.y = 0;

          that.set_speed(speed);
          that.set_acceleration(accel);
        }
    }
} // collision_event::execute()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the other item has crossed the surface following a top to down
 *        direction.
 * \param info Informations on the collision.
 * \param self The first item in the collision.
 * \param that The other item in the collision.
 * \param line The line representing the top of the current item.
 */
bool bear::collision_event_slope::item_crossed_up_down
( const universe::collision_info& info, universe::physical_item& self,
  universe::physical_item& that, const line_type& line ) const
{
  bool result = false;

  if ( that.get_bottom() >= line.y_value(that.get_center_of_mass().x) )
    {
      const universe::position_type other_prev_bottom
        ( info.other_previous_state().get_center_of_mass().x,
          info.other_previous_state().get_bottom() );
      const universe::coordinate_type y_left = 
        self.get_position().y + m_surface.origin.y;
      const universe::coordinate_type y_right = 
        line.y_value( self.get_right() );


      if ( other_prev_bottom.x < self.get_position().x )
        result = other_prev_bottom.y <= y_left + m_surface_width;
      else if ( other_prev_bottom.x > self.get_right() )
        result = other_prev_bottom.y <= y_right + m_surface_width;
      else
        result = other_prev_bottom.y
          <= line.y_value(other_prev_bottom.x) + m_surface_width;
    }

  return result;
} // collision_event_slope::item_crossed_up_down()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the other item has crossed the surface following a top to down
 *        direction.
 * \param info Informations on the collision.
 * \param self The first item in the collision.
 * \param that The other item in the collision.
 * \param line The line representing the top of the current item.
 */
bool bear::collision_event_slope::item_crossed_down_up
( const universe::collision_info& info, universe::physical_item& self,
  universe::physical_item& that, const line_type& line ) const
{
  bool result = false;

  if ( that.get_position().y <= line.y_value(that.get_center_of_mass().x) )
    {
      const universe::position_type other_prev_top
        ( info.other_previous_state().get_center_of_mass().x,
          info.other_previous_state().get_position().y );
      const universe::coordinate_type y_right =
        self.get_position().y + m_surface.origin.y;
      const universe::coordinate_type y_left = 
        line.y_value( self.get_position().x );

      if ( other_prev_top.x < self.get_position().x )
        result = other_prev_top.y >= y_left - m_surface_width;
      else if ( other_prev_top.x > self.get_right() )
        result = other_prev_top.y >= y_right - m_surface_width;
      else
        result = other_prev_top.y >= line.y_value(other_prev_top.x);
    }

  return result;
} // collision_event_slope::item_crossed_down_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Align the other item above the current item.
 * \param that The other item in the collision.
 * \param line The line representing the top of the current item.
 */
void bear::collision_event_slope::align_on_top
( universe::physical_item& that, universe::physical_item& self,
  const claw::math::line_2d<universe::coordinate_type>& line ) const
{
  const universe::position_type pos
    ( that.get_center_of_mass().x, line.y_value(that.get_center_of_mass().x) );

  that.set_bottom_middle(pos);
  that.set_bottom_contact(self);
  self.set_top_contact(that);

  universe::speed_type speed( that.get_speed() );

  if ( (m_surface.direction.y > 0) && (speed.x < 0)
       || (m_surface.direction.y < 0) && (speed.x > 0) )
    if ( line.direction.x != 0 )
      {
        universe::speed_type::value_type ratio =
          (universe::speed_type::value_type)std::abs(line.direction.y)
          / ( 2 * (universe::speed_type::value_type)line.direction.x );

        speed.x *= 1.0 - 
          std::min(ratio, universe::speed_type::value_type(1.0));
        that.set_speed(speed);
      }
} // collision_event_slope::align_on_top()

/*----------------------------------------------------------------------------*/
/**
 * \brief Align the other item below the current item.
 * \param that The other item in the collision.
 * \param line The line representing the bottom of the current item.
 */
void bear::collision_event_slope::align_below
( universe::physical_item& that, universe::physical_item& self,
  const claw::math::line_2d<universe::coordinate_type>& line ) const
{
  const universe::position_type pos
    ( that.get_center_of_mass().x, line.y_value(that.get_center_of_mass().x) );

  that.set_top_middle(pos);
  that.set_top_contact(self);
  self.set_bottom_contact(that);
} // collision_event_slope::align_on_top()


/*----------------------------------------------------------------------------*/
/**
 * \brief Replace the other item above the current item.
 * \param info Informations on the collision.
 * \param self The first item in the collision.
 * \param that The other item in the collision.
 * \param line The line representing the top of the current item.
 */
void bear::collision_event_slope::align_replace
( const universe::collision_info& info, 
  universe::physical_item& that, universe::physical_item& self,
  const claw::math::line_2d<universe::coordinate_type>& line ) const
{
  const claw::math::line_2d<universe::coordinate_type> move_that
    ( info.other_previous_state().get_bottom_middle(), 
      that.get_bottom_middle() - 
      info.other_previous_state().get_bottom_middle() );

  //  const claw::math::line_2d<double> new_line(line);

  //const claw::math::coordinate_2d<double> new_pos 
  //  = move_that.intersection(new_line);
  //const universe::position_type pos(new_pos);
  
  const universe::position_type pos = move_that.intersection(line);
  
  /*
    std::cout << "-----------" << std::endl;
    std::cout << move_that.origin.x << " ; " << move_that.origin.y << std::endl;
    std::cout << move_that.direction.x << " ; " 
    << move_that.direction.y << std::endl;
    std::cout << line.origin.x << " ; " << line.origin.y << std::endl;
    std::cout << line.direction.x << " ; " << line.direction.y << std::endl;
    std::cout << pos.x << " ; " << pos.y << std::endl;
  */
    
  that.set_bottom_middle(pos);
  that.set_bottom_contact(self);
  self.set_top_contact(that);

  universe::speed_type speed( that.get_speed() );

  if ( (m_surface.direction.y > 0) && (speed.x < 0)
       || (m_surface.direction.y < 0) && (speed.x > 0) )
    if ( line.direction.x != 0 )
      {
        universe::speed_type::value_type ratio =
          (universe::speed_type::value_type)std::abs(line.direction.y)
          / ( 2 * (universe::speed_type::value_type)line.direction.x );

        speed.x *= 1.0 - std::min(ratio, universe::speed_type::value_type(1.0));
        that.set_speed(speed);
      }
} // collision_event_slope::align_replace()
