/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file collision_event_slope.hpp
 * \brief A collision event that align an item on a slope.
 * \author Julien Jorge
 */
#ifndef __BEAR_COLLISION_EVENT_SLOPE_HPP__
#define __BEAR_COLLISION_EVENT_SLOPE_HPP__

#include "universe/collision_event/collision_event.hpp"

namespace bear
{
  /**
   * \brief A collision event that align an item on a slope.
   *
   * The item is aligned if the x-coodinate of its center of mass is in the
   * interval covered by the current item.
   *
   * \author Julien Jorge
   */
  class collision_event_slope : public universe::collision_event
  {
  public:
    /** \brief The type of the line describing the surface of the slope. */
    typedef claw::math::line_2d<universe::coordinate_type> line_type;

  public:
    collision_event_slope( const line_type& surface, 
                           universe::coordinate_type line_width );

    void execute( const universe::collision_info& info,
                  universe::physical_item& self,
                  universe::physical_item& that ) const;

  private:
    bool item_crossed_up_down
    ( const universe::collision_info& info, universe::physical_item& self,
      universe::physical_item& that, const line_type& line ) const;
    bool item_crossed_down_up
    ( const universe::collision_info& info, universe::physical_item& self,
      universe::physical_item& that, const line_type& line ) const;

    void
    align_on_top( universe::physical_item& that, universe::physical_item& self,
		  const line_type& line ) const;
    void
    align_below( universe::physical_item& that, universe::physical_item& self,
		 const line_type& line ) const;

    void align_replace( const universe::collision_info& info, 
			universe::physical_item& that, 
			universe::physical_item& self,
			const claw::math::line_2d<universe::coordinate_type>& line ) const;

  private:
    /** \brief The line describing the surface of the slope. */
    const line_type m_surface;

    /** \brief The width of the line: defines the distance above/below the line
        in which the collision is checked when the collision occurs. */
    const universe::coordinate_type m_surface_width;

  }; // class collision_event_slope
} // namespace bear

#endif // __BEAR_COLLISION_EVENT_SLOPE_HPP__
