/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file slope.hpp
 * \brief A ground that is not horizontal.
 * \author Julien Jorge
 */
#ifndef __BEAR_INVISIBLE_SLOPE_HPP__
#define __BEAR_INVISIBLE_SLOPE_HPP__

#include "engine/base_item.hpp"

#include "generic_items/collision_event/collision_event_slope.hpp"
#include "universe/collision_event/collision_event_align_stop.hpp"
#include "universe/collision_event/conditional_collision_event.hpp"
#include "universe/collision_test/bottom_contact_is_lower.hpp"

namespace bear
{
  /**
   * \brief A invisible ground that is not horizontal.
   *
   * The parameters accepted by this item are:
   *  - \a steepness: \c real The difference beween the Y-coordinate of the
   *    invisible_slope's right and left edges,
   *  - \a is_ground: \c boolean Tell is the invisible_slope is a ground
   *    (otherwise it is a ceiling),
   *  - \a opposite_side_is_solid: \c boolean Turn on/off the alignement on the
   *    side opposite to the invisible_slope,
   *  - \a left_side_is_solid: \c boolean Turn on/off the alignement on the left
   *    side,
   *  - \a right_side_is_solid: \c boolean Turn on/off the alignement on the
   *    right side,
   *  - any field supported by the parent class.
   *
   * - A ground with a positive steepness starts from the top-left corner.
   * - A ground with a negative steepness starts from the bottom-left corner.
   * - A ceiling with a positive steepness starts from the bottom-right corner.
   * - A ceiling with a negative steepness starts from the top-right corner.
   *
   * \author Julien Jorge
   */
  class invisible_slope :
    public engine::base_item
  {
  public:
    /** \brief The type of the parent class. */
    typedef engine::base_item super;

  private:
    typedef universe::collision_event_align_stop<universe::align_top>
    ce_align_stop_top;

    typedef universe::collision_event_align_stop<universe::align_left>
    ce_align_stop_left;

    typedef universe::collision_event_align_stop<universe::align_right>
    ce_align_stop_right;

    typedef universe::collision_event_align_stop<universe::align_bottom>
    ce_align_stop_bottom;

    typedef
    universe::conditional_collision_event<universe::bottom_contact_is_lower>
    ce_condition_bottom_lower;

  public:
    invisible_slope();

    bool set_real_field( const std::string& name, double value );
    bool set_bool_field( const std::string& name, bool value );

    bool is_valid() const;

    void start();

  private:
    void set_ground_down();
    void set_ground_up();
    void set_ceiling_down();
    void set_ceiling_up();

    void create_middle_ground_collision_event
    ( const collision_event_slope::line_type& line );

  private:
    /** \brief Is it a ground or a ceiling ? */
    bool m_is_ground;

    /** \brief The steepness of the invisible_slope. */
    universe::coordinate_type m_steepness;

    /** \brief The side opposite to the invisible_slope, is it solid ? */
    bool m_opposite_side_is_solid;

    /** \brief Tell if the left side of the block is solid. */
    bool m_left_side_is_solid;

    /** \brief Tell if the right side of the block is solid. */
    bool m_right_side_is_solid;

    /** \brief The width of the surface of the invisible_slope. */
    static const universe::coordinate_type s_line_width;

    collision_event_slope::line_type m_line;

  }; // class invisible_slope
} // namespace bear

#endif // __BEAR_INVISIBLE_SLOPE_HPP__
