/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file action_layer.tpp
 * \brief Implementation of the template methods of the
 *        bear::action_layer class.
 * \author Julien Jorge
 */
#include "engine/level_border.hpp"
#include "engine/level.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Create vertical margin in the level.
 * \param height The height of the blocks.
 * \param x The X-coordinate of the blocks.
 */
template<typename Event>
void bear::action_layer::create_vertical_margin
( universe::size_type height, universe::coordinate_type x )
{
  const universe::size_type max_y =
    get_size().y - engine::level::s_level_margin;

  universe::coordinate_type y = engine::level::s_level_margin;
  
  for ( ; y < max_y - height; y += height )
    new_margin_block<Event>( x, y, engine::level::s_level_margin, height );

  new_margin_block<Event>
    ( x, y, engine::level::s_level_margin, max_y - y );
} // action_layer::create_vertical_margin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create horizontal margin in the level.
 * \param width The width of the blocks.
 * \param y The Y-coordinate of the blocks.
 */
template<typename Event>
void bear::action_layer::create_horizontal_margin
( universe::size_type width, universe::coordinate_type y )
{
  const universe::size_type max_x =
    get_size().x - engine::level::s_level_margin;

  universe::coordinate_type x = engine::level::s_level_margin;
  
  for ( ; x < max_x - width; x += width )
    new_margin_block<Event>( x, y, width, engine::level::s_level_margin );

  new_margin_block<Event>
    ( x, y, max_x - x, engine::level::s_level_margin );
} // action_layer::create_horizontal_margin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a block for the margins.
 * \param x X-coordinate of the block.
 * \param y Y-coordinate of the block.
 * \param width The width of the block.
 * \param height The height of the block.
 */
template<typename Event>
void bear::action_layer::new_margin_block
( universe::coordinate_type x, universe::coordinate_type y,
  universe::size_type width, universe::size_type height )
{
  typedef engine::level_border<Event> block_type;

  block_type* block = new block_type;

  block->set_position( x, y );
  block->set_real_field( "width", width );
  block->set_real_field( "height", height );
  block->fix();

  CLAW_ASSERT( block->is_valid(),
               "action_layer::new_margin_block(): the block is not valid" );

  add_item( *block );
} // action_layer::new_margin_block()
