/*
  Bear Engine

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file decoration_layer.cpp
 * \brief Implementation of the bear::decoration_layer class.
 * \author Julien Jorge
 */
#include "generic_items/layer/decoration_layer.hpp"

#include "engine/export.hpp"

#include <claw/logger.hpp>
#include <set>

LAYER_EXPORT( decoration_layer, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the bounding box of an item.
 * \param item The concerned item.
 */
claw::math::rectangle<unsigned int>
bear::decoration_layer::item_traits::get_bounding_box
( engine::base_item* const& item ) const
{
  CLAW_PRECOND( item != NULL );

  return item->get_bounding_box().cast_value_type_to<unsigned int>();
} // decoration_layer::item_traits::get_bounding_box()




/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param world_size The real size of the world.
 */
bear::decoration_layer::decoration_layer
( const universe::size_box_type& size )
  : layer( size ),
    m_items( (unsigned int)m_size.x + 1, (unsigned int)m_size.y + 1, 256 )
{

} // decoration_layer::decoration_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::decoration_layer::~decoration_layer()
{
  std::set<engine::base_item*> items;
  std::set<engine::base_item*>::const_iterator it;

  m_items.get_all(items);

  for (it=items.begin(); it!=items.end(); ++it)
    delete *it;
} // decoration_layer::~decoration_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an item in the layer.
 * \param item The item to add. It will be deleted with the layer.
 */
void bear::decoration_layer::add_item( engine::base_item& item )
{
  m_items.insert( &item );
} // decoration_layer::add_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start the items in the layer.
 */
void bear::decoration_layer::start()
{
  std::set<engine::base_item*> items;

  m_items.get_all( items );

  std::set<engine::base_item*>::const_iterator it;

  for (it=items.begin(); it!=items.end(); ++it)
    (*it)->start();

  log_statistics();
} // decoration_layer::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Progress the items in the active area.
 * \param active_area The area containing active items.
 * \param elapsed_time Elapsed time since the last call.
 */
void bear::decoration_layer::progress
( const region_type& active_area, universe::time_type elapsed_time  )
{
  std::set<engine::base_item*> items;
  region_type::const_iterator it;

  for (it=active_area.begin(); it!=active_area.end(); ++it)
    m_items.get_area( it->cast_value_type_to<unsigned int>(), items );

  std::set<engine::base_item*>::const_iterator it_i;

  for (it_i=items.begin(); it_i!=items.end(); ++it_i)
    (*it_i)->progress( elapsed_time );
} // decoration_layer::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the visible sprites of this layer.
 * \param visuals (out) The sprites in the visible area, and their positions.
 * \param visible_area The visible part of the layer.
 */
void bear::decoration_layer::get_visual
( std::list<engine::scene_visual>& visuals,
  const universe::rectangle_type& visible_area ) const
{
  std::set<engine::base_item*> items;

  m_items.get_area( visible_area.cast_value_type_to<unsigned int>(), items );

  std::set<engine::base_item*>::const_iterator it;

  for (it=items.begin(); it!=items.end(); ++it)
    (*it)->get_visual( visuals );
} // decoration_layer::get_visual()

/*----------------------------------------------------------------------------*/
/**
 * \brief Log layer's inside maps load.
 */
void bear::decoration_layer::log_statistics() const
{
  unsigned int empty_cells, min, max;
  double avg;

  empty_cells = m_items.empty_cells();
  m_items.cells_load(min, max, avg);

  claw::logger << claw::log_verbose << "layer[" << m_size.x << ":" << m_size.y
               << "]:items:empty=" << empty_cells << " min=" << min
               << " max=" << max << " avg=" << avg << claw::lendl;
} // decoration_layer::log_statistics()
