/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/config_frame.cpp
 * \brief Implementation of the bf::config_frame class.
 * \author Julien Jorge
 */
#include "bf/config_frame.hpp"

#include "bf/wx_facilities.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param parent The window owning this dialog.
 * \param config The configuration of the program.
 */
bf::config_frame::config_frame( wxWindow* parent, configuration& config )
  : wxDialog( parent, wxID_ANY, _("Configuration") ), m_config(config)
{
  create_controls();
  Fit();
} // config_frame::config_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Fill the controls with the values of the config.
 */
void bf::config_frame::fill_controls()
{
  m_item_classes_text->SetValue
    ( std_to_wx_string( m_config.item_class_path ) );
  m_data_path_text->SetValue( std_to_wx_string( m_config.data_path ) );
} // config_frame::fill_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls of our frame.
 */
void bf::config_frame::create_controls()
{
  create_member_controls();
  create_sizer_controls();
  fill_controls();
} // config_frame::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls for which we keep a reference.
 */
void bf::config_frame::create_member_controls()
{
  m_item_classes_text =
    new wxTextCtrl
    ( this, wxID_ANY, wxT(""), wxDefaultPosition, wxSize(400, -1) );
  m_data_path_text =
    new wxTextCtrl
    ( this, wxID_ANY, wxT(""), wxDefaultPosition, wxSize(400, -1) );
} // config_frame::create_member_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls and add them in sizers.
 */
void bf::config_frame::create_sizer_controls()
{
  wxBoxSizer* sizer = new wxBoxSizer( wxVERTICAL );

  // item classes
  wxStaticBoxSizer* s_sizer = new wxStaticBoxSizer
    ( wxHORIZONTAL, this, _("Path to the item classes files") );

  s_sizer->Add( m_item_classes_text, 1, wxEXPAND | wxALL, 5 );
  s_sizer->Add
    ( new wxButton( this, IDC_BROWSE_ITEM_CLASSES_BUTTON, wxT("..."),
                    wxDefaultPosition, wxSize(30, -1) ), 0, wxALL, 5 );

  sizer->Add( s_sizer, 0, wxEXPAND );

  // item classes
  s_sizer = new wxStaticBoxSizer
    ( wxHORIZONTAL, this, _("Path to data directory of the game") );

  s_sizer->Add( m_data_path_text, 1, wxEXPAND | wxALL, 5 );
  s_sizer->Add
    ( new wxButton( this, IDC_BROWSE_DATA_PATH_BUTTON, wxT("..."),
                    wxDefaultPosition, wxSize(30, -1) ), 0, wxALL, 5 );

  sizer->Add( s_sizer, 0, wxEXPAND );

  // ok / cancel
  sizer->Add( CreateStdDialogButtonSizer(wxOK | wxCANCEL), 0,
              wxALL | wxCENTER, 5 );

  SetSizer(sizer);
} // config_frame::create_sizer_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the "Ok" button.
 * \param event This event occured.
 */
void bf::config_frame::on_ok(wxCommandEvent& event)
{
  m_config.item_class_path =
    wx_to_std_string( m_item_classes_text->GetValue() );
  m_config.data_path = wx_to_std_string( m_data_path_text->GetValue() );

  m_config.save();

  EndModal(wxID_OK);
} // config_frame::on_ok()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the "Cancel" button.
 * \param event This event occured.
 */
void bf::config_frame::on_cancel(wxCommandEvent& event)
{
  fill_controls();

  EndModal(wxID_CANCEL);
} // config_frame::on_cancel()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the "Browse" button for the item files.
 * \param event This event occured.
 */
void bf::config_frame::on_browse_item_classes(wxCommandEvent& event)
{
  wxDirDialog diag(this);

  if ( diag.ShowModal() == wxID_OK )
    m_item_classes_text->SetValue( diag.GetPath() );
} // config_frame::on_browse_item_classes()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on the "Browse" button for the data directory.
 * \param event This event occured.
 */
void bf::config_frame::on_browse_data_path(wxCommandEvent& event)
{
  wxDirDialog diag(this);

  if ( diag.ShowModal() == wxID_OK )
    m_data_path_text->SetValue( diag.GetPath() );
} // config_frame::on_browse_data_path()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::config_frame, wxDialog)
  EVT_BUTTON( wxID_OK, bf::config_frame::on_ok )
  EVT_BUTTON( wxID_CANCEL, bf::config_frame::on_cancel )
  EVT_BUTTON( bf::config_frame::IDC_BROWSE_ITEM_CLASSES_BUTTON,
              bf::config_frame::on_browse_item_classes )
  EVT_BUTTON( bf::config_frame::IDC_BROWSE_DATA_PATH_BUTTON,
              bf::config_frame::on_browse_data_path )
END_EVENT_TABLE()
