/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/level.cpp
 * \brief Implementation of the bf::level class.
 * \author Julien Jorge
 */
#include "bf/level.hpp"

#include "level_code_value.hpp"
#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param width The width of the level.
 * \param height The height of the level.
 * \param cam_w The width of the camera.
 * \param cam_h The height of the camera.
 * \param mus The music played in this level.
 */
bf::level::level( unsigned int width, unsigned int height, unsigned int cam_w,
                  unsigned int cam_h, const std::string& mus )
  : m_width(width), m_height(height), m_camera_width(cam_w),
    m_camera_height(cam_h), m_music(mus)
{

} // level::level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 */
bf::level::level( const level& that )
{
  assign(that);
} // level::level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bf::level::~level()
{
  clear();
} // level::~level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a layer given its position in the stack.
 * \param index The position of the layer.
 */
bf::layer& bf::level::get_layer( unsigned int index )
{
  CLAW_PRECOND( index < m_layer.size() );
  return *m_layer[index];
} // level::get_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a layer given its position in the stack.
 * \param index The position of the layer.
 */
const bf::layer& bf::level::get_layer( unsigned int index ) const
{
  CLAW_PRECOND( index < m_layer.size() );
  return *m_layer[index];
} // level::get_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove a layer given its position in the stack.
 * \param index The position of the layer.
 */
void bf::level::remove_layer( unsigned int index )
{
  CLAW_PRECOND( index < m_layer.size() );

  delete m_layer[index];
  m_layer.erase(m_layer.begin() + index);
} // level::remove_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a layer.
 * \param width The width of the layer.
 * \param height The height of the layer.
 * \param class_name The name of the class of the layer.
 */
bf::layer& bf::level::add_layer
( unsigned int width, unsigned int height, const std::string& class_name )
{
  m_layer.push_back( new layer( width, height, class_name ) );
  return *m_layer.back();
} // level::add_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move a layer toward the background.
 * \param layer_index The position of the layer.
 */
void bf::level::move_backward( unsigned int layer_index )
{
  if ( layer_index > 0 )
    std::swap( m_layer[layer_index], m_layer[layer_index-1] );
} // level::move_backward()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move a layer toward the foreground.
 * \param layer_index The position of the layer.
 */
void bf::level::move_forward( unsigned int layer_index )
{
  if ( layer_index + 1 < m_layer.size() )
    std::swap( m_layer[layer_index], m_layer[layer_index+1] );
} // level::move_forward()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the number of layers.
 */
unsigned int bf::level::layers_count() const
{
  return m_layer.size();
} // level::layers_count()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the width of the level.
 */
unsigned int bf::level::get_width() const
{
  return m_width;
} // level::get_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the height of the level.
 */
unsigned int bf::level::get_height() const
{
  return m_height;
} // level::get_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the width of the camera.
 */
unsigned int bf::level::get_camera_width() const
{
  return m_camera_width;
} // level::get_camera_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the height of the camera.
 */
unsigned int bf::level::get_camera_height() const
{
  return m_camera_height;
} // level::get_camera_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the music of the level.
 */
const std::string& bf::level::get_music() const
{
  return m_music;
} // level::get_music()

/*----------------------------------------------------------------------------*/
/**
 * \brief Find an item with a given id.
 * \param id The id of the searched item.
 */
std::pair<bool, bf::layer::const_item_iterator>
bf::level::find_item_by_id( const std::string& id ) const
{
  bool found = false;
  layer::const_item_iterator it;

  for (unsigned int i=0; !found && (i!=m_layer.size()); ++i)
    {
      layer::const_item_iterator eit( ((const layer*)m_layer[i])->item_end() );
      it=((const layer*)m_layer[i])->item_begin();

      while ( !found && (it!=eit) )
        if ( it->get_id() == id )
          found = true;
        else
          ++it;
    }

  return std::pair<bool, layer::const_item_iterator>(found, it);
} // level::find_item_by_id()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the configuration of the level.
 * \param width The width of the level.
 * \param height The height of the level.
 * \param cam_w The width of the camera.
 * \param cam_h The height of the camera.
 * \param mus The music played in this level.
 */
void bf::level::set
( unsigned int width, unsigned int height, unsigned int cam_w,
  unsigned int cam_h, const std::string& mus )
{
  m_width = width;
  m_height = height;
  m_camera_width = cam_w;
  m_camera_height = cam_h;
  m_music = mus;
} // level::set()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assignment operator.
 * \param that The instance to copy from.
 */
bf::level& bf::level::operator=( const level& that )
{
  if ( &that != this )
    {
      clear();
      assign(that);
    }

  return *this;
} // level::operator=()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if the level is valid.
 * Return the bad item (NULL if level is valid).
 * \param layer_error The indice of bad layer.
 * \param error_msg The error message.
 */
bf::item_instance*
bf::level::check( unsigned int& layer_error, std::string& error_msg )
{
  item_instance* item = NULL;
  std::set<std::string> list_id;
  
  // check unicity of identifiant
  for (unsigned int i=0; ( i!=m_layer.size() ) && ( item == NULL ); ++i)
    {
      layer::item_iterator it;
      for (it=get_layer(i).item_begin(); (it!=get_layer(i).item_end()); ++it)
	if ( ! it->get_id().empty() )
	  if ( list_id.find(it->get_id()) != list_id.end() )
	    {
	      error_msg = "Error item " + it->get_class_name() + 
		" : the identifier '" + it->get_id() + 
		"' has been already given";
	      layer_error = i;
	      item = &(*it);
	    }
	  else
	    list_id.insert(it->get_id());      
    }

  // check each item
  for (unsigned int i=0; ( i!=m_layer.size() ) && ( item == NULL ); ++i)
    {
      item = get_layer(i).check(error_msg);
      if ( item != NULL )
	layer_error = i;
    }

  return item;
} // level::check()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the level.
 * \param f The file in which we compile.
 */
void bf::level::compile( compiled_file& f ) const
{
  f << m_width << m_height << m_camera_width << m_camera_height << m_music
    << m_layer.size();

  for (unsigned int i=0; i!=m_layer.size(); ++i)
    {
      f << bear::level_code_value::layer;
      m_layer[i]->compile(f);
    }

  f << bear::level_code_value::eof;
} // level::compile()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove all layers.
 */
void bf::level::clear()
{
  for (unsigned int i=0; i!=m_layer.size(); ++i)
    delete m_layer[i];

  m_layer.clear();
} // level::clear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assign a level to this one.
 * \param that The instance to copy from.
 */
void bf::level::assign( const level& that )
{
  CLAW_PRECOND( m_layer.empty() );
  CLAW_PRECOND( &that != this );

  m_width = that.m_width;
  m_height = that.m_height;
  m_camera_width = that.m_camera_width;
  m_camera_height = that.m_camera_height;
  m_music = that.m_music;

  m_layer.resize( that.m_layer.size() );

  for (unsigned int i=0; i!=that.m_layer.size(); ++i)
    m_layer[i] = new layer( *that.m_layer[i] );
} // level::assign()
