/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/level_file_xml_reader.cpp
 * \brief Implementation of the bf::level_file_xml_reader class.
 * \author Julien Jorge
 */
#include "bf/level_file_xml_reader.hpp"

#include "bf/item_class_pool.hpp"
#include "bf/wx_facilities.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Read the value from a xml value node.
 * \param spr (out) The sprite we have read.
 * \param node The node from which we read the value.
 */
void bf::xml_to_value<bf::sprite>::operator()
  ( sprite& spr, wxXmlNode* node ) const
{
  CLAW_PRECOND( node != NULL );

  wxString image_name;

  if ( !node->GetPropVal( wxT("image"), &image_name ) )
    throw xml_missing_property("image");

  spr.set_image_name( wx_to_std_string(image_name) );

  spr.set_left( level_file_xml_reader::read_uint(node, "x") );
  spr.set_top( level_file_xml_reader::read_uint(node, "y") );
  spr.set_clip_width( level_file_xml_reader::read_uint(node, "clip_width") );
  spr.set_clip_height( level_file_xml_reader::read_uint(node, "clip_height") );
  spr.set_width
    ( level_file_xml_reader::read_uint_opt
      (node, "width", spr.get_clip_width()) );
  spr.set_height
    ( level_file_xml_reader::read_uint_opt
      (node, "height", spr.get_clip_height()) );
  spr.set_flip_x( level_file_xml_reader::read_bool_opt(node, "flip_x", false));
  spr.set_flip_y( level_file_xml_reader::read_bool_opt(node, "flip_y", false));
  spr.set_alpha( level_file_xml_reader::read_real_opt(node, "alpha", 1) );
} // xml_to_value::operator()() [sprite]

/*----------------------------------------------------------------------------*/
/**
 * \brief Read the value from a xml value node.
 * \param anim (out) The animation we have read.
 * \param node The node from which we read the value.
 */
void bf::xml_to_value<bf::animation>::operator()
  ( animation& anim, wxXmlNode* node ) const
{
  CLAW_PRECOND( node != NULL );

  anim.set_flip_x
    ( level_file_xml_reader::read_bool_opt(node, "flip_x", false));
  anim.set_flip_y
    ( level_file_xml_reader::read_bool_opt(node, "flip_y", false));
  anim.set_alpha
    ( level_file_xml_reader::read_real_opt(node, "alpha", 1) );
  anim.set_loops( level_file_xml_reader::read_uint(node, "loops") );
  anim.set_first_index
    ( level_file_xml_reader::read_uint(node, "first_index") );
  anim.set_last_index
    ( level_file_xml_reader::read_uint(node, "last_index") );
  anim.set_loop_back
    ( level_file_xml_reader::read_bool_opt(node, "loop_back", false));
      
  load_frames(anim, node->GetChildren());
} // xml_to_value::operator()() [animation]

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the frames of an animation.
 * \param anim The animation in which we set a list of frames.
 * \param node The node to parse.
 */
void bf::xml_to_value<bf::animation>::load_frames
( animation& anim, wxXmlNode* node ) const
{
  for ( ; node!=NULL; node=node->GetNext() )
    if ( node->GetName() == wxT("frame") )
      load_frame(anim, node);
    else
      claw::logger << claw::log_warning << "Ignored node '"
                   << wx_to_std_string(node->GetName()) << "'" << claw::lendl;
} // bf::xml_to_value<bf::animation>::load_frames()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load a frame of an animation.
 * \param anim The animation in which we add the frame.
 * \param node The node to parse.
 */
void bf::xml_to_value<bf::animation>::load_frame
( animation& anim, wxXmlNode* node ) const
{
  CLAW_PRECOND( node->GetName() == wxT("frame") );

  wxString val;
  animation::frame frame;
  sprite spr;

  if ( !node->GetPropVal( wxT("duration"), &val ) )
    throw xml_missing_property( "duration" );
  
  frame.set_duration
    ( level_file_xml_reader::read_real_opt(node, "duration",40) );
  
  wxXmlNode* children = node->GetChildren();
  if ( children != NULL )
    {
      if ( children->GetName() == wxT("sprite") )
        {
          xml_to_value<sprite> xml_conv;       
          xml_conv(spr,children);
          frame.set_sprite(spr);
          anim.get_frames().push_back(frame);
        }
      else
        claw::logger << claw::log_warning << "Ignored node '"
                     << wx_to_std_string(children->GetName()) 
                     << "'" << claw::lendl;      
    }
  else
    throw xml_missing_node("sprite");
} // bf::xml_to_value<bf::animation>::load_frame()




/*----------------------------------------------------------------------------*/
/**
 * \brief Load a level.
 * \param file_path The path to the level file.
 */
bf::level* bf::level_file_xml_reader::load( const wxString& file_path ) const
{
  wxXmlDocument doc;

  if ( !doc.Load(file_path) )
    throw std::ios_base::failure
      ( "Cannot load the XML file '" + wx_to_std_string(file_path) + "'" );

  wxXmlNode* node = doc.GetRoot();

  if ( node == NULL )
    throw xml_missing_node("level");

  return load_level( node );
} // level_file_xml_reader::load()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert the value of a property in an unsigned integer.
 * \param node The node in which we take the property.
 * \param prop The name of the property to get.
 */
unsigned int
bf::level_file_xml_reader::read_uint( wxXmlNode* node, const std::string& prop )
{
  unsigned int result;
  wxString val;

  if ( !node->GetPropVal( std_to_wx_string(prop), &val ) )
    throw xml_missing_property(prop);

  std::istringstream iss( wx_to_std_string(val) );

  if ( !(iss >> result) )
    throw xml_bad_value( "unsigned integer", wx_to_std_string(val) );

  return result;
} // level_file_xml_reader::read_uint()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert the value of a property in an unsigned integer.
 * \param node The node in which we take the property.
 * \param prop The name of the property to get.
 */
unsigned int bf::level_file_xml_reader::read_uint_opt
( wxXmlNode* node, const std::string& prop, unsigned int def )
{
  unsigned int result(def);
  wxString val;

  if ( node->GetPropVal( std_to_wx_string(prop), &val ) )
    {
      std::istringstream iss( wx_to_std_string(val) );
      iss >> result;
    }

  return result;
} // level_file_xml_reader::read_uint_opt()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert the value of an optional property in a real.
 * \param node The node in which we take the property.
 * \param prop The name of the property to get.
 * \param def The default value.
 */
double bf::level_file_xml_reader::read_real_opt
( wxXmlNode* node, const std::string& prop, double def )
{
  double result(def);
  wxString val;

  if ( node->GetPropVal( std_to_wx_string(prop), &val ) )
    {
      std::istringstream iss( wx_to_std_string(val) );
      iss >> result;
    }

  return result;
} // level_file_xml_reader::read_real_opt()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert the value of an optional property in a boolean.
 * \param node The node in which we take the property.
 * \param prop The name of the property to get.
 * \param def The default value.
 */
bool bf::level_file_xml_reader::read_bool_opt
( wxXmlNode* node, const std::string& prop, bool def )
{
  bool result(def);
  wxString val;

  if ( node->GetPropVal( std_to_wx_string(prop), &val ) )
    {
      if ( val == wxT("true") )
	result = true;
      else if ( val == wxT("false") )
	result = false;
    }
  
  return result;
} // level_file_xml_reader::read_bool_opt()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load a node of type "level".
 * \param node The node to parse.
 */
bf::level* bf::level_file_xml_reader::load_level( wxXmlNode* node ) const
{
  if ( node->GetName() != wxT("level") )
    throw xml_bad_node( wx_to_std_string(node->GetName()) );

  unsigned int width, height, cam_w, cam_h;
  wxString val;

  width = read_uint(node, "width");
  height = read_uint(node, "height");
  cam_w = read_uint(node, "camera_width");
  cam_h = read_uint(node, "camera_height");
  
  val = node->GetPropVal( wxT("music"), wxT("") );

  level* lvl = new level( width, height, cam_w, cam_h, wx_to_std_string(val) );

  load_layers( *lvl, node->GetChildren() );

  return lvl;
} // level_file_xml_reader::load_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the layers of the level.
 * \param lvl The level in which we store the layers.
 * \param node The node to parse.
 */
void bf::level_file_xml_reader::load_layers( level& lvl, wxXmlNode* node ) const
{
  for ( ; node!=NULL; node=node->GetNext() )
    if ( node->GetName() == wxT("layer") )
      load_layer(lvl, node);
    else
      claw::logger << claw::log_warning << "Ignored node '"
                   << wx_to_std_string(node->GetName()) << "'" << claw::lendl;
} // level_file_xml_reader::load_layers()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load one layer.
 * \param lvl The level in which we store the layer.
 * \param node The node to parse.
 */
void bf::level_file_xml_reader::load_layer( level& lvl, wxXmlNode* node ) const
{
  CLAW_PRECOND( node->GetName() == wxT("layer") );

  unsigned int width, height;
  wxString val;

  width = read_uint(node, "width");
  height = read_uint(node, "height");
  
  if ( !node->GetPropVal( wxT("class_name"), &val ) )
    throw xml_missing_property( "class_name" );

  layer& lay = lvl.add_layer( width, height, wx_to_std_string(val) );

  load_layer_content( lay, node->GetChildren() );
} // level_file_xml_reader::load_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the content of a layer.
 * \param lay The layer in which we store the instances.
 * \param node The node to parse.
 */
void bf::level_file_xml_reader::load_layer_content
( layer& lay, wxXmlNode* node ) const
{
  for ( ; node!=NULL; node=node->GetNext() )
    if ( node->GetName() == wxT("item") )
      load_item(lay, node);
    else
      claw::logger << claw::log_warning << "Ignored node '"
                   << wx_to_std_string(node->GetName()) << "'" << claw::lendl;
} // level_file_xml_reader::load_layer_content()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load an item of a layer.
 * \param lay The layer in which we store the item.
 * \param node The node to parse.
 */
void bf::level_file_xml_reader::load_item( layer& lay, wxXmlNode* node ) const
{
  CLAW_PRECOND( node->GetName() == wxT("item") );

  wxString val;

  if ( !node->GetPropVal( wxT("class_name"), &val ) )
    throw xml_missing_property( "class_name" );

  std::string class_name( wx_to_std_string(val) );

  if ( !item_class_pool::get_instance().has_item_class(class_name) )
    claw::logger << claw::log_warning << "Unknown item class '" << class_name
                 << "'" << claw::lendl;
  else
    {
      item_instance item( class_name );
      item.set_fixed( read_bool_opt(node, "fixed", false) );
      item.set_id( wx_to_std_string(node->GetPropVal( wxT("id"), wxT("") )) );

      load_fields( item, node->GetChildren() );

      lay.add_item(item);
    }
} // level_file_xml_reader::load_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the fields of an item.
 * \param item The item in which we set the fields.
 * \param node The node to parse.
 */
void bf::level_file_xml_reader::load_fields
( item_instance& item, wxXmlNode* node ) const
{
  for ( ; node!=NULL; node=node->GetNext() )
    if ( node->GetName() == wxT("field") )
      load_field(item, node);
    else
      claw::logger << claw::log_warning << "Ignored node '"
                   << wx_to_std_string(node->GetName()) << "'" << claw::lendl;
} // level_file_xml_reader::load_fields()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load a field of an item.
 * \param item The item in which we set the field.
 * \param node The node to parse.
 */
void bf::level_file_xml_reader::load_field
( item_instance& item, wxXmlNode* node ) const
{
  CLAW_PRECOND( node->GetName() == wxT("field") );

  wxString val;

  if ( !node->GetPropVal( wxT("name"), &val ) )
    throw xml_missing_property( "name" );

  std::string field_name( wx_to_std_string(val) );

  const item_class& the_class =
    item_class_pool::get_instance().get_item_class( item.get_class_name() );

  if ( !the_class.has_field(field_name) )
    claw::logger << claw::log_warning << "Unknown field '" << field_name << "'"
                 << claw::lendl;
  else
    {
      const type_field& field = the_class.get_field(field_name);
      load_field( item, field, node->GetChildren() );
    }
} // level_file_xml_reader::load_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the value of a field.
 * \param item The item in which we set the field.
 * \param f The field to load.
 * \param node The node to parse.
 */
void bf::level_file_xml_reader::load_field
( item_instance& item, const type_field& f, wxXmlNode* node ) const
{
  if ( f.is_list() )
    switch ( f.get_field_type() )
      {
      case type_field::integer_field_type:
        load_value_list<integer_type>( item, f.get_name(), "integer", node );
        break;
      case type_field::u_integer_field_type:
        load_value_list<u_integer_type>
          ( item, f.get_name(), "u_integer", node );
        break;
      case type_field::real_field_type:
        load_value_list<real_type>( item, f.get_name(), "real", node );
        break;
      case type_field::boolean_field_type:
        load_value_list<bool_type>( item, f.get_name(), "bool", node );
        break;
      case type_field::string_field_type:
        load_value_list<string_type>( item, f.get_name(), "string", node );
        break;
      case type_field::sprite_field_type:
        load_value_list<sprite>( item, f.get_name(), "sprite", node );
        break;
      case type_field::animation_field_type:
        load_value_list<animation>( item, f.get_name(), "animation", node );
        break;
      case type_field::item_reference_field_type:
        load_value_list<item_reference_type>
          ( item, f.get_name(), "item_reference", node );
        break;
      }
  else
    switch ( f.get_field_type() )
      {
      case type_field::integer_field_type:
        load_value<integer_type>( item, f.get_name(), "integer", node );
        break;
      case type_field::u_integer_field_type:
        load_value<u_integer_type>( item, f.get_name(), "u_integer", node );
        break;
      case type_field::real_field_type:
        load_value<real_type>( item, f.get_name(), "real", node );
        break;
      case type_field::boolean_field_type:
        load_value<bool_type>( item, f.get_name(), "bool", node );
        break;
      case type_field::string_field_type:
        load_value<string_type>( item, f.get_name(), "string", node );
        break;
      case type_field::sprite_field_type:
        load_value<sprite>( item, f.get_name(), "sprite", node );
        break;
      case type_field::animation_field_type:
        load_value<animation>( item, f.get_name(), "animation", node );
        break;
      case type_field::item_reference_field_type:
        load_value<item_reference_type>
          ( item, f.get_name(), "item_reference", node );
        break;
      }
} // level_file_xml_reader::load_field()
