/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/code/level_file_xml_writer.cpp
 * \brief Implementation of the bf::level_file_xml_writer class.
 * \author Julien Jorge
 */
#include "bf/level_file_xml_writer.hpp"

#include "bf/item_class_pool.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Save a level.
 * \param os The stream in which we write.
 * \param lvl The level to save.
 */
void bf::level_file_xml_writer::save( std::ostream& os, const level& lvl ) const
{
  os << "<level width='" << lvl.get_width() << "' height='" << lvl.get_height()
     << "' camera_width='" << lvl.get_camera_width() << "' camera_height='"
     << lvl.get_camera_height() << "' music='" << lvl.get_music() << "'>\n";

  for (unsigned int i=0; i!=lvl.layers_count(); ++i)
    save_layer( os, lvl.get_layer(i) );

  os << "</level>\n";
} // level_file_xml_writer::save()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save a layer.
 * \param os The stream in which we write.
 * \param the_layer The layer to save.
 */
void bf::level_file_xml_writer::save_layer
( std::ostream& os, const layer& the_layer ) const
{
  os << "  <layer class_name='" << the_layer.get_class_name() << "' width='"
     << the_layer.get_width() << "' height='" << the_layer.get_height()
     << "'>\n\n";

  layer::const_item_iterator it;

  for (it=the_layer.item_begin(); it!=the_layer.item_end(); ++it)
    save_item( os, *it );

  os << "  </layer><!-- " << the_layer.get_class_name() << " -->\n\n";
} // level_file_xml_writer::save_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save an item.
 * \param os The stream in which we write.
 * \param item The item to save.
 */
void bf::level_file_xml_writer::save_item
( std::ostream& os, const item_instance& item ) const
{
  os << "    <item class_name='" << item.get_class_name() << "' fixed='";

  if ( item.get_fixed() )
    os << "true'";
  else
    os << "false'";

  if ( !item.get_id().empty() )
    os << " id='" << item.get_id() << "'";

  os << ">\n";

  save_item_by_class( os, item, item.get_class_name() );

  os << "    </item><!-- " << item.get_class_name() << " -->\n\n";
} // level_file_xml_writer::save_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the fields of an item by exploring its parent classes.
 * \param os The stream in which we write.
 * \param item The item to save.
 * \param class_name The name of the class from which we start.
 */
void bf::level_file_xml_writer::save_item_by_class
( std::ostream& os, const item_instance& item,
  const std::string& class_name ) const
{
  const item_class& the_class =
    item_class_pool::get_instance().get_item_class( class_name );

  const std::list<std::string>& super_class = the_class.get_super_classes();
  std::list<std::string>::const_iterator itc;

  for ( itc=super_class.begin(); itc!=super_class.end(); ++itc )
    save_item_by_class(os, item, *itc);

  item_class::field_iterator it;

  for ( it=the_class.field_begin(); it!=the_class.field_end(); ++it )
    if ( item.has_value(*it) )
      {
        os << "      <field name='" << it->get_name() << "'>\n";
        save_field( os, *it, item );
        os << "      </field>\n";
      }
} // level_file_xml_writer::save_item_by_class()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field.
 * \param os The stream in which we write.
 * \param f The field to save.
 * \param item The item in which we take the field.
 */
void bf::level_file_xml_writer::save_field
( std::ostream& os, const type_field& f, const item_instance& item ) const
{
  if ( f.is_list() )
    switch ( f.get_field_type() )
      {
      case type_field::integer_field_type:
        save_value_list<integer_type>( os, f.get_name(), item, "integer" );
        break;
      case type_field::u_integer_field_type:
        save_value_list<u_integer_type>( os, f.get_name(), item, "u_integer" );
        break;
      case type_field::real_field_type:
        save_value_list<real_type>( os, f.get_name(), item, "real" );
        break;
      case type_field::boolean_field_type:
        save_value_list<bool_type>( os, f.get_name(), item, "bool" );
        break;
      case type_field::string_field_type:
        save_value_list<string_type>( os, f.get_name(), item, "string" );
        break;
      case type_field::sprite_field_type:
        save_sprite_list( os, f.get_name(), item );
        break;
      case type_field::animation_field_type:
        save_animation_list( os, f.get_name(), item );
        break;
      case type_field::item_reference_field_type:
        save_value_list<item_reference_type>
          ( os, f.get_name(), item, "item_reference" );
        break;
      }
  else
    switch ( f.get_field_type() )
      {
      case type_field::integer_field_type:
        save_value<integer_type>( os, f.get_name(), item, "integer" );
        break;
      case type_field::u_integer_field_type:
        save_value<u_integer_type>( os, f.get_name(), item, "u_integer" );
        break;
      case type_field::real_field_type:
        save_value<real_type>( os, f.get_name(), item, "real" );
        break;
      case type_field::boolean_field_type:
        save_value<bool_type>( os, f.get_name(), item, "bool" );
        break;
      case type_field::string_field_type:
        save_value<string_type>( os, f.get_name(), item, "string" );
        break;
      case type_field::sprite_field_type:
        save_sprite( os, f.get_name(), item );
        break;
      case type_field::animation_field_type:
        save_animation( os, f.get_name(), item );
        break;
      case type_field::item_reference_field_type:
        save_value<item_reference_type>
          ( os, f.get_name(), item, "item_reference" );
        break;
      }
} // level_file_xml_writer::save_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of type 'sprite'.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 */
void bf::level_file_xml_writer::save_sprite
( std::ostream& os, const std::string& field_name,
  const item_instance& item ) const
{
  sprite spr;
  item.get_value( field_name, spr );
  sprite_to_xml(os, spr);
} // level_file_xml_writer::save_sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of type 'animation'.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 */
void bf::level_file_xml_writer::save_animation
( std::ostream& os, const std::string& field_name,
  const item_instance& item ) const
{
  animation anim;
  item.get_value( field_name, anim );
  animation_to_xml(os, anim);
} // level_file_xml_writer::save_animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of type 'list of sprite'.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 */
void bf::level_file_xml_writer::save_sprite_list
( std::ostream& os, const std::string& field_name,
  const item_instance& item ) const
{
  std::list<sprite> spr;
  std::list<sprite>::const_iterator it;

  item.get_value( field_name, spr );

  for (it=spr.begin(); it!=spr.end(); ++it)
    sprite_to_xml(os, *it);
} // level_file_xml_writer::save_sprite_list()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of type 'list of animation'.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 */
void bf::level_file_xml_writer::save_animation_list
( std::ostream& os, const std::string& field_name,
  const item_instance& item ) const
{
  std::list<animation> anim;
  std::list<animation>::const_iterator it;

  item.get_value( field_name, anim );

  for (it=anim.begin(); it!=anim.end(); ++it)
    animation_to_xml(os, *it);
} // level_file_xml_writer::save_animation_list()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create a node for a sprite.
 * \param os The stream in which we save the value.
 * \param spr The sprite to write.
 */
void bf::level_file_xml_writer::sprite_to_xml
( std::ostream& os, const sprite& spr ) const
{
  os << "        <sprite image='" << spr.get_image_name() << "' x='"
     << spr.get_left() << "' y='" << spr.get_top() << "' clip_width='"
     << spr.get_clip_width() << "' clip_height='" 
     << spr.get_clip_height() << "' width='"
     << spr.get_width() << "' height='" << spr.get_height() << "' flip_x='";

  if ( spr.get_flip_x() )
    os << "true";
  else
    os << "false";

  os << "' flip_y='" ;

  if ( spr.get_flip_y() )
    os << "true";
  else
    os << "false";

 os << "' alpha='"
     << spr.get_alpha() << "'/>\n";
} // level_file_xml_writer::sprite_to_xml()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create a node for a animation.
 * \param os The stream in which we save the value.
 * \param anim The animation to write.
 */
void bf::level_file_xml_writer::animation_to_xml
( std::ostream& os, const animation& anim ) const
{
  os << "        <animation flip_x='";
  if ( anim.get_flip_x() )
    os << "true";
  else
    os << "false";

  os << "' flip_y='" ;
  if ( anim.get_flip_y() )
    os << "true";
  else
    os << "false";

  os << "' alpha='" << anim.get_alpha() << "' loops='" << anim.get_loops() << 
    "' first_index='" <<  anim.get_first_index() << "' last_index='" << 
    anim.get_last_index();

  os << "' loop_back='" ;
  if ( anim.get_loop_back() )
    os << "true'>\n";
  else
    os << "false'>\n";

  std::list<animation::frame>::const_iterator it;

  for ( it = anim.get_frames().begin(); it != anim.get_frames().end(); ++it )
    {
      os << "          <frame duration='" << it->get_duration() << "'>\n";
      os << "    ";
      sprite_to_xml(os,it->get_sprite());
      os << "          </frame>\n";
    }  

  os << "        </animation>\n";
} // level_file_xml_writer::animation_to_xml()
