/*
    Plee The Bear - Level editor

    Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file bf/ingame_view.hpp
 * \brief This control display the items of the level.
 * \author Julien Jorge
 */
#ifndef __BF_INGAME_VIEW_HPP__
#define __BF_INGAME_VIEW_HPP__

#include <wx/wx.h>
#include <wx/dnd.h>

#include "bf/level_clipboard.hpp"
#include "bf/level_history.hpp"
#include "bf/windows_layout.hpp"
#include "bf/grid.hpp"

namespace bf
{
  class ingame_view_frame;

  /**
   * \brief This control display the items of the level.
   * \author Julien Jorge
   */
  class ingame_view:
    public wxWindow
  {
  public:
    typedef wxWindow super;

  private:
    /** \brief Informations on a drag local to the view. */
    class drag_info
    {
    public:
      enum drag_mode_type
        {
          drag_mode_none,
          drag_mode_selection,
          drag_mode_move,
          drag_mode_size
        }; // enum drag_mode

    public:
      drag_info();

      wxSize delta() const;

    public:
      /** \brief The current mode of the drag. */
      drag_mode_type drag_mode;

      /** \brief The item picked at the begining of the drag. */
      item_instance* picked_item;

      /** \brief Origin of the drag. */
      wxPoint mouse_origin;

      /** \brief Position of the mouse during the drag. */
      wxPoint mouse_position;

    }; // struct drag_info

    /** \brief A drop target for creating an item. The class of the item is
        passed as a simple text. */
    class item_drop_target:
      public wxTextDropTarget
    {
    public:
      item_drop_target( ingame_view& view );

      bool OnDropText(wxCoord x, wxCoord y, const wxString& data);

    private:
      /** \brief The view associated with this target. */
      ingame_view& m_view;

    }; // class item_drop_target

  public:
    ingame_view
    ( ingame_view_frame& parent, level* lvl, windows_layout& layout );

    bool empty() const;

    const wxPoint& get_view_position() const;
    void set_view_position( wxCoord x, wxCoord y );
    
    unsigned int get_active_index() const;
    void set_active_index( unsigned int index );

    const layer& get_active_layer() const;
    const level& get_level() const;
    level& get_level();

    bool get_visible( unsigned int index );
    void set_visible( unsigned int index, bool vis );

    void show_grid( bool v );
    bool grid_visible() const;

    const grid& get_grid() const;
    grid& get_grid();
    void set_grid( const grid& g );

    void add_layer
    ( unsigned int width, unsigned int height, const std::string& class_name );
    void remove_layer();
    void change_layer
    ( unsigned int x, unsigned int y, const std::string& class_name );

    void move_backward();
    void move_forward();

    bool add_item( const std::string& class_name, wxCoord x, wxCoord y );
    bool add_item( const std::string& class_name );

    void save( std::ostream& os );
    bool revert_save();
    void put_level(level* lvl);

    void start_change();
    void undo();
    void redo();
    const level_history& get_history() const;

    void update_layout();

    bool has_selection() const;
    void delete_selection();
    void clear_selection();
    void select_all();
    void select_item_and_layer( item_instance* item,
				const unsigned int index );

    const level_clipboard& get_clipboard() const;

    void copy_to_clipboard() const;
    void cut_to_clipboard();
    void paste_from_clipboard(bool add_to_selection = false);

    wxBitmap create_thumbnail( const wxSize& s ) const;
    void render();

  private:
    void render_layers( wxDC& dc ) const;
    void render_layer( wxDC& dc, unsigned int i ) const;
    void render_layer_full( wxDC& dc, unsigned int i ) const;

    void
    render_item( wxDC& dc, const item_instance& item, bool active ) const;
    void render_item
    ( wxDC& dc, const item_instance& item, const wxPoint& pos,
      bool active ) const;
    void render_item_as_point
    ( wxDC& dc, const item_instance& item, const wxPoint& pos,
      bool active ) const;
    void render_item_as_box
    ( wxDC& dc, const item_instance& item, const wxPoint& pos,
      const wxSize& size, bool active ) const;
    bool render_sprite
    ( wxDC& dc, const item_instance& item, const wxPoint& pos ) const;
    void render_item_bounding_box
    ( wxDC& dc, const item_instance& item, const wxPoint& pos,
      const wxSize& size, bool active ) const;

    void render_grip( wxDC& dc ) const;
    void render_drag( wxDC& dc ) const;
    void render_drag_mode_selection( wxDC& dc ) const;
    void render_drag_mode_move( wxDC& dc ) const;
    void render_drag_mode_size( wxDC& dc ) const;
    void render_layer_background( wxDC& dc ) const;
    void render_grid( wxDC& dc ) const;

    std::pair<wxBitmap, wxPoint> get_item_visual
    ( const item_instance& item, const wxPoint& pos ) const;
    wxPen get_display_pen( const item_instance& item, bool active ) const;

    layer& current_layer();

    item_instance* pick_item( const wxPoint& pos );
    void pick_item( std::list<item_instance*>& item, const wxRect& box );
    void toggle_selection( item_instance* item );
    void add_selection( item_instance* item );
    void add_selection( const std::list<item_instance*>& item);
    void set_selection( const std::list<item_instance*>& item);
    void set_selection
    ( const std::list<item_instance*>& item, item_instance* selected,
      bool add_selection = false);
    void set_selection( item_instance* item );

    wxRect get_bounding_box( const item_instance& item ) const;
    wxRect get_presence_box( const item_instance& item ) const;

    void copy_selection(bool add_selection = false);
    void move_selection();
    
    void toggle_graphic_drawing();
    void toggle_wireframe_drawing();
    int update_coordinate_magnetism
    ( unsigned int item_position, unsigned int size_item,
      unsigned int offset_grid, unsigned int step_grid);
    void update_mouse_position( const wxPoint& mouse_position );

    void move_pixel_selection(int keycode);
    void move_grid(int keycode);
    void move_view(int keycode);

    void write_mouse_position(const wxPoint& point);

    void next_layer();
    void previous_layer();

    void set_drag_mode_size( const wxPoint& pos );
    wxRect get_drag_mode_size_box() const;

    void apply_drag_mode_move( bool ctrl, bool shift, bool alt );
    void apply_drag_mode_selection( bool ctrl, bool alt );
    void apply_drag_mode_size();

    void on_size(wxSizeEvent& event);
    void on_paint(wxPaintEvent& event);
    void on_mouse_left_down(wxMouseEvent& event);
    void on_mouse_move(wxMouseEvent& event);
    void on_mouse_left_up(wxMouseEvent& event);
    void on_mouse_wheel_rotation(wxMouseEvent& event);
    void on_key_down(wxKeyEvent& event);
    void on_key_up(wxKeyEvent& event);
    
  private:
    /** \brief The frame containing the view. */
    ingame_view_frame& m_parent;

    /** \brief The windows of the interface. */
    windows_layout& m_layout;

    /** \brief The level and the undo/redo informations. */
    level_history m_history;

    /** \brief Position of the view in the level. */
    wxPoint m_view;

    /** \brief Index of the layer on which we are working. */
    unsigned int m_active_layer;

    /** \brief The item currently selected. */
    item_instance* m_selected_item;

    /** \brief The items is a multiple selection. */
    std::set<item_instance*> m_selected_group;

    /** \brief Info about the current dragging. */
    drag_info* m_drag_info;

    /** \brief Tell, for each layer, if it is visible or not. */
    std::vector<bool> m_visible;

    /** \brief The grid. */
    grid m_grid;

    /** \brief Tell if we draw the bounding box of the items. */
    bool m_wireframe_drawing;

    /** \brief Tell if we display the pictures of the items. */
    bool m_graphic_drawing;

    /** \brief Tell if we display the grid. */
    bool m_display_grid;

    /** \brief The size of the shape diplayed for sizeless items. */
    static const wxCoord s_point_size;

    /** \brief The size of grip in the corner of the selected item. */
    static const wxCoord s_grip_size;

    /** \brief The items copied in the clipboard. */
    static level_clipboard s_clipboard;

    DECLARE_EVENT_TABLE()

  }; // class ingame_view
} // namespace bf

#endif // __BF_INGAME_VIEW_HPP__
