/*
  Plee The Bear - Model compiler

  Copyright (C) 2005-2008 Julien Jorge, Sébastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file model_grammar.tpp
 * \brief Implemenation of the mc::model_grammar class.
 * \author Julien Jorge.
 */

#include <boost/spirit/tree/ast.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Default constructor.
 */
template <typename ScannerT>
mc::model_grammar::definition<ScannerT>::
error_report_parser::error_report_parser()
  : m_msg("")
{

} // definition::error_report_parser::error_report_parser()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param msg A text explaining the error.
 */
template <typename ScannerT>
mc::model_grammar::definition<ScannerT>::
error_report_parser::error_report_parser( const std::string msg )
  : m_msg(msg)
{

} // definition::error_report_parser::error_report_parser()

/*----------------------------------------------------------------------------*/
/**
 * \brief Print the message on standard error output.
 * \param scan The scanner detecting the error.
 */
template <typename ScannerT>
template <typename LocalScanner>
int
mc::model_grammar::definition<ScannerT>::error_report_parser::operator()
  (const LocalScanner& scan, result_t& /*result*/) const
{
  bs::file_position fpos = scan.first.get_position();

  std::cerr << fpos.file << ":" << fpos.line <<":" << fpos.column << ": "
            << m_msg << std::endl;

  return -1;
} // definition::operator()

/*----------------------------------------------------------------------------*/
/**
 * \brief Default constructor.
 */
template <typename ScannerT>
mc::model_grammar::definition<ScannerT>::
char_error_report_parser::char_error_report_parser()
{

} // definition::char_error_report_parser::char_error_report_parser()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param c The missing character.
 */
template <typename ScannerT>
mc::model_grammar::definition<ScannerT>::
char_error_report_parser::char_error_report_parser( char c )
  : error_report_parser( std::string("Missing character '") + c + "'." )
{

} // definition::char_error_report_parser::char_error_report_parser()

/*----------------------------------------------------------------------------*/
/**
 * \brief Default constructor.
 */
template <typename ScannerT>
mc::model_grammar::definition<ScannerT>::
keyword_error_report_parser::keyword_error_report_parser()
{

} // definition::keyword_error_report_parser::keyword_error_report_parser()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param keyword The missing keyword.
 */
template <typename ScannerT>
mc::model_grammar::definition<ScannerT>::
keyword_error_report_parser::keyword_error_report_parser
( const std::string& keyword )
  : error_report_parser( "Keyword '" + keyword + "' expected." )
{

} // definition::keyword_error_report_parser::keyword_error_report_parser()

/*----------------------------------------------------------------------------*/
/**
 * \brief Default constructor.
 */
template <typename ScannerT>
mc::model_grammar::definition<ScannerT>::
type_error_report_parser::type_error_report_parser()
{

} // definition::type_error_report_parser::type_error_report_parser()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param type The expected type.
 */
template <typename ScannerT>
mc::model_grammar::definition<ScannerT>::
type_error_report_parser::type_error_report_parser( const std::string& type )
  : error_report_parser( "Type '" + type + "' expected." )
{

} // definition::type_error_report_parser::type_error_report_parser()


/*----------------------------------------------------------------------------*/
/**
 * \brief Definition of the rules.
 * \param self The concerned mc::model_grammar instance.
 */
template<typename ScannerT>
mc::model_grammar::definition<ScannerT>::definition
( const model_grammar& self )
{
  initialize_error_parsers();

  // Whole file
  m_file =
    ( m_resources_part | m_error_resources_part )
    >> !m_model_part;

  define_miscellanious();
  define_types();
  define_resources_part();
  define_model_part();
} // model_grammar::definition::definition()

/*----------------------------------------------------------------------------*/
/**
 * \brief Definition of miscellanious rules.
 */
template<typename ScannerT>
void mc::model_grammar::definition<ScannerT>::define_miscellanious()
{
  m_identifier =
    bs::token_node_d
    [ bs::lexeme_d[ ( bs::alpha_p
                      | '_'
                      )
                    >> *( bs::alnum_p
                          | '_'
                          ) ]
      ]
    ;

  m_filename =
    bs::lexeme_d
    [ bs::no_node_d[ bs::ch_p('"') ]
      >> bs::token_node_d[ *( ~bs::ch_p('"')
                              & ~bs::space_p ) ]
      >> bs::no_node_d[ ( '"' | m_error_not_terminated_string ) ]
      ]
    ;

} // model_grammar::definition::define_miscellanious()

/*----------------------------------------------------------------------------*/
/**
 * \brief Definition of the rules describing the types.
 */
template<typename ScannerT>
void mc::model_grammar::definition<ScannerT>::define_types()
{
  m_any_type =
    m_image_type
    | m_animation_type
    | m_bool_type
    | bs::longest_d[ m_u_integer_type
                     | m_real_type ]
    | m_string_type
    | m_identifier
    ;

  m_u_integer_type =
    ( bs::no_node_d[ bs::str_p("u_integer") ]
      >> bs::no_node_d[ ( '(' | m_error_left_parenthesis ) ]
      >> ( m_u_integer_type   | m_error_u_integer )
      >> bs::no_node_d[ ( ')' | m_error_right_parenthesis ) ] )
    | bs::uint_p
    ;

  m_real_type =
    ( bs::no_node_d[ bs::str_p("real") ]
      >> bs::no_node_d[ ( '(' | m_error_left_parenthesis ) ]
      >> ( m_real_type        | m_error_real )
      >> bs::no_node_d[ ( ')' | m_error_right_parenthesis ) ] )
    | bs::real_p
    ;

  m_bool_type =
    ( bs::no_node_d[ bs::str_p("bool") ]
      >> bs::no_node_d[ ( '(' | m_error_left_parenthesis ) ]
      >> ( m_bool_type        | m_error_bool )
      >> bs::no_node_d[ ( ')' | m_error_right_parenthesis ) ] )
    | bs::token_node_d[ bs::str_p("true") ]
    | bs::token_node_d[ bs::str_p("false") ]
    ;

  m_string_type =
    ( bs::no_node_d[ bs::str_p("string") ]
      >> bs::inner_node_d[ ( '('              | m_error_left_parenthesis )
                           >> ( m_string_type | m_error_string )
                           >> ( ')'           | m_error_right_parenthesis ) ] )
    | ( bs::lexeme_d
        [ bs::no_node_d[ bs::ch_p('"') ]
          >> bs::token_node_d[ *( ~bs::ch_p('"')
                                  & ~bs::space_p ) ]
          >> bs::no_node_d[ ( '"' | m_error_not_terminated_string ) ]
          ] )
    ;

  m_image_type =
    bs::no_node_d[ bs::str_p("image")
                   >> ( '('                    | m_error_left_parenthesis ) ]
    >> bs::infix_node_d[ ( m_identifier        | m_error_identifier )
                         >> ( ','              | m_error_comma )
                         >> ( m_u_integer_type | m_error_u_integer )
                         >> ( ','              | m_error_comma )
                         >> ( m_u_integer_type | m_error_u_integer )
                         >> ( ','              | m_error_comma )
                         >> ( m_u_integer_type | m_error_u_integer )
                         >> ( ','              | m_error_comma )
                         >> ( m_u_integer_type | m_error_u_integer )
                         >> ( ')'              | m_error_right_parenthesis ) ]
    >> !( bs::no_node_d[ bs::ch_p('{') ]
          >> *m_image_body_item
          >> bs::no_node_d[ ( '}'              | m_error_right_brace ) ] )
    ;

  m_animation_type =
    bs::no_node_d[ bs::str_p("animation")
                   >> ( '(' | m_error_left_parenthesis ) ]
    >> m_frame_list
    >> bs::no_node_d[ ( ')' | m_error_right_parenthesis ) ]
    >> !( bs::no_node_d[ bs::ch_p('{') ]
          >> *m_image_body_item
          >> bs::no_node_d[ ( '}' | m_error_right_brace ) ] )
    ;

  m_image_body_item =
    bs::infix_node_d[ ( m_identifier
                        >> ( '='        | m_error_equal )
                        >> ( m_any_type | m_error_type )
                        >> ( ';'        | m_error_semicolon ) ) ]
    | ( ~bs::ch_p('}') >> m_error_identifier )
    ;

  m_frame_list =
    bs::no_node_d[ ( '[' | m_error_left_bracket ) ]
    >> bs::infix_node_d[ ( m_image_type
                           | m_identifier
                           | m_error_image )
                         >> *( ( ',' | ( ~bs::ch_p(']') >> m_error_comma ) )
                               >> ( m_image_type
                                    | m_identifier
                                    | m_error_image )
                               )
                         >> ( ']'          | m_error_right_bracket ) ]
    ;
} // model_grammar::definition::define_types()

/*----------------------------------------------------------------------------*/
/**
 * \brief Definition of the rules for the resources part.
 */
template<typename ScannerT>
void mc::model_grammar::definition<ScannerT>::define_resources_part()
{
  m_resources_part =
    bs::no_node_d[bs::str_p("resources")]
    >> bs::no_node_d[ bs::ch_p('{') | m_error_left_brace ]
    >> *m_resource
    >> bs::no_node_d[ bs::ch_p('}') | m_error_right_brace ]
    ;

  m_resource =
    ( m_resource_image
      | m_resource_sound
      | ( ~bs::ch_p('}') >> m_error_resource ) )
    >> bs::no_node_d[ ( ';' | m_error_semicolon ) ]
    ;

  m_resource_image =
    bs::no_node_d[ bs::str_p("gfx")
                   >> ( '('                | m_error_left_parenthesis ) ]
    >> bs::infix_node_d[ ( m_identifier    | m_error_identifier )
                         >> ( ','          | m_error_comma )
                         >> ( m_filename   | m_error_filename )
                         >> ( ')'          | m_error_right_parenthesis ) ]
    ;

  m_resource_sound =
    bs::no_node_d[ bs::str_p("sound")
                   >> ( '('                | m_error_left_parenthesis ) ]
    >> bs::infix_node_d[ ( m_identifier    | m_error_identifier )
                         >> ( ','          | m_error_comma )
                         >> ( m_filename   | m_error_filename )
                         >> ( ')'          | m_error_right_parenthesis ) ]
    ;
} // model_grammar::definition::define_resources_part()

/*----------------------------------------------------------------------------*/
/**
 * \brief Definition of the rules for the model part.
 */
template<typename ScannerT>
void mc::model_grammar::definition<ScannerT>::define_model_part()
{
  // Model part
  m_model_part =
    bs::no_node_d[ bs::str_p("model") ]
    >> bs::no_node_d[ ( bs::ch_p('{')     | m_error_left_brace ) ]
    >> m_model_description
    >> bs::no_node_d[ ( '}'               | m_error_right_brace ) ]
    ;

  m_model_description =
    *( m_model_atom
       >> bs::no_node_d[ ( ';' | m_error_semicolon ) ] )
    ;

  m_model_atom =
    m_const_declaration
    | m_action
    | ( ~bs::ch_p('}') >> m_error_model_atom )
    ;

  m_const_declaration =
    bs::no_node_d[ bs::str_p("let") ]
    >> ( m_identifier       | m_error_identifier )
    >> bs::no_node_d[ ( '=' | m_error_equal ) ]
    >> ( m_any_type         | m_error_type )
    ;

  m_action =
    ( bs::no_node_d[ bs::str_p("action") ]
      >> bs::infix_node_d[ m_identifier
                           >> ( '('               | m_error_left_parenthesis )
                           >> ( m_identifier
                                | m_animation_type
                                | m_error_animation )
                           >> ( ','           | m_error_comma )
                           >> ( m_identifier  | m_error_identifier )
                           >> ( ')'           | m_error_right_parenthesis ) ] )
    | ( ~bs::ch_p('}') >> m_error_identifier )
    ;
} // model_grammar::definition::define_model_part()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the error parsers
 */
template<typename ScannerT>
void mc::model_grammar::definition<ScannerT>::initialize_error_parsers()
{
  m_error_identifier = error_report_parser( "Identifier expected." );
  m_error_filename = error_report_parser( "Filename expected." );
  m_error_animation = error_report_parser( "Animation expected." );
  m_error_type = error_report_parser( "Type or identifier expected." );
  m_error_not_terminated_string =
    error_report_parser( "Not terminated string." );
  m_error_resource = error_report_parser( "'gfx' or 'sound' expected." );
  m_error_model_atom =
    error_report_parser( "Expected 'let' or 'action'." );

  m_error_resources_part = keyword_error_report_parser( "resources" );

  m_error_colon = char_error_report_parser( ':' );
  m_error_comma = char_error_report_parser( ',' );
  m_error_equal = char_error_report_parser( '=' );
  m_error_semicolon = char_error_report_parser( ';' );
  m_error_right_bracket = char_error_report_parser( ']' );
  m_error_left_bracket = char_error_report_parser( '[' );
  m_error_right_parenthesis = char_error_report_parser( ')' );
  m_error_left_parenthesis = char_error_report_parser( '(' );
  m_error_right_brace = char_error_report_parser( '}' );
  m_error_left_brace = char_error_report_parser( '{' );

  m_error_u_integer = type_error_report_parser( "u_integer" );
  m_error_image = type_error_report_parser( "image" );
  m_error_bool = type_error_report_parser( "bool" );
  m_error_real = type_error_report_parser( "real" );
  m_error_string = type_error_report_parser( "string" );

} // model_grammar::definition::initialize_error_parsers()
