/*
  Plee The Bear - Model compiler

  Copyright (C) 2005-2008 Julien Jorge, Sébastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file symbol_table.cpp
 * \brief Implementation of the mc::symbol_table class.
 * \author Julien Jorge
 */
#include "mc/symbol_table.hpp"

#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove all symbols from the table.
 */
void mc::symbol_table::clear()
{
  m_u_integer.clear();
  m_real.clear();
  m_bool.clear();
  m_string.clear();
  m_image.clear();
  m_animation.clear();
} // symbol_table::clear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a constant of type unsigned integer.
 * \param name The name of the constant.
 * \param value The value of the constant.
 * \pre !constant_exists(name)
 */
void mc::symbol_table::add_u_integer( const std::string& name,
                                      const u_integer_type& value )
{
  CLAW_PRECOND( !constant_exists(name) );

  m_u_integer[name] = value;
} // symbol_table::add_u_integer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a constant of type real.
 * \param name The name of the constant.
 * \param value The value of the constant.
 * \pre !constant_exists(name)
 */
void mc::symbol_table::add_real( const std::string& name,
                                 const real_type& value )
{
  CLAW_PRECOND( !constant_exists(name) );

  m_real[name] = value;
} // symbol_table::add_real()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a constant of type bool.
 * \param name The name of the constant.
 * \param value The value of the constant.
 * \pre !constant_exists(name)
 */
void mc::symbol_table::add_bool( const std::string& name,
                                 const bool_type& value )
{
  CLAW_PRECOND( !constant_exists(name) );

  m_bool[name] = value;
} // symbol_table::add_bool()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a constant of type string.
 * \param name The name of the constant.
 * \param value The value of the constant.
 * \pre !constant_exists(name)
 */
void mc::symbol_table::add_string( const std::string& name,
                                   const string_type& value )
{
  CLAW_PRECOND( !constant_exists(name) );

  m_string[name] = value;
} // symbol_table::add_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a constant of type image.
 * \param name The name of the constant.
 * \param value The value of the constant.
 * \pre !constant_exists(name)
 */
void mc::symbol_table::add_image( const std::string& name,
                                  const image_type& value )
{
  CLAW_PRECOND( !constant_exists(name) );

  m_image[name] = value;
} // symbol_table::add_image()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a constant of type animation.
 * \param name The name of the constant.
 * \param value The value of the constant.
 * \pre !constant_exists(name)
 */
void mc::symbol_table::add_animation( const std::string& name,
                                      const animation_type& value )
{
  CLAW_PRECOND( !constant_exists(name) );

  m_animation[name] = value;
} // symbol_table::add_animation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there's a constant with a specific name.
 * \param name The name of the constant.
 */
bool mc::symbol_table::constant_exists( const std::string& name ) const
{
  return u_integer_exists(name)
    || real_exists(name)
    || bool_exists(name)
    || string_exists(name)
    || image_exists(name)
    || animation_exists(name);
} // symbol_table::constant_exists()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there's a constant of type unsigned integer with a specific
 *        name.
 * \param name The name of the constant.
 */
bool mc::symbol_table::u_integer_exists( const std::string& name ) const
{
  return m_u_integer.find(name) != m_u_integer.end();
} // symbol_table::u_integer_exists()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there's a constant of type real with a specific name.
 * \param name The name of the constant.
 */
bool mc::symbol_table::real_exists( const std::string& name ) const
{
  return m_real.find(name) != m_real.end();
} // symbol_table::real_exists()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there's a constant of type bool with a specific name.
 * \param name The name of the constant.
 */
bool mc::symbol_table::bool_exists( const std::string& name ) const
{
  return m_bool.find(name) != m_bool.end();
} // symbol_table::bool_exists()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there's a constant of type string with a specific name.
 * \param name The name of the constant.
 */
bool mc::symbol_table::string_exists( const std::string& name ) const
{
  return m_string.find(name) != m_string.end();
} // symbol_table::string_exists()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there's a constant of type image with a specific name.
 * \param name The name of the constant.
 */
bool mc::symbol_table::image_exists( const std::string& name ) const
{
  return m_image.find(name) != m_image.end();
} // symbol_table::image_exists()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there's a constant of type animation with a specific name.
 * \param name The name of the constant.
 */
bool mc::symbol_table::animation_exists( const std::string& name ) const
{
  return m_animation.find(name) != m_animation.end();
} // symbol_table::animation_exists()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a constant of type u_integer.
 * \param name The name of the constant.
 * \pre u_integer_exists(name)
 */
const mc::u_integer_type&
mc::symbol_table::get_u_integer( const std::string& name ) const
{
  CLAW_PRECOND( u_integer_exists(name) );

  return m_u_integer.find(name)->second;
} // symbol_table::get_u_integer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a constant of type real.
 * \param name The name of the constant.
 * \pre real_exists(name)
 */
const mc::real_type&
mc::symbol_table::get_real( const std::string& name ) const
{
  CLAW_PRECOND( real_exists(name) );

  return m_real.find(name)->second;
} // symbol_table::get_real()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a constant of type bool.
 * \param name The name of the constant.
 * \pre bool_exists(name)
 */
const mc::bool_type&
mc::symbol_table::get_bool( const std::string& name ) const
{
  CLAW_PRECOND( bool_exists(name) );

  return m_bool.find(name)->second;
} // symbol_table::get_bool()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a constant of type string.
 * \param name The name of the constant.
 * \pre string_exists(name)
 */
const mc::string_type&
mc::symbol_table::get_string( const std::string& name ) const
{
  CLAW_PRECOND( string_exists(name) );

  return m_string.find(name)->second;
} // symbol_table::get_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a constant of type image.
 * \param name The name of the constant.
 * \pre image_exists(name)
 */
const mc::image_type&
mc::symbol_table::get_image( const std::string& name ) const
{
  CLAW_PRECOND( image_exists(name) );

  return m_image.find(name)->second;
} // symbol_table::get_image()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a constant of type animation.
 * \param name The name of the constant.
 * \pre animation_exists(name)
 */
const mc::animation_type&
mc::symbol_table::get_animation( const std::string& name ) const
{
  CLAW_PRECOND( animation_exists(name) );

  return m_animation.find(name)->second;
} // symbol_table::get_animation()
