/*
  Plee The Bear - Model compiler

  Copyright (C) 2005-2008 Julien Jorge, Sébastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file node_compiler_image_type.cpp
 * \brief Implementation of the mc::node_compiler_image_type class.
 * \author Julien Jorge
 */
#include "mc/node_compiler/node_compiler_image_type.hpp"

#include <claw/assert.hpp>
#include "mc/model_grammar.hpp"

#include "mc/node_compiler/node_compiler_identifier.hpp"
#include "mc/node_compiler/node_compiler_u_integer_type.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile a node of type image_type.
 * \param val The value read.
 * \param symbols The symbols of the model.
 * \param the_model The model we are compiling.
 * \param node Node to compile.
 */
void mc::node_compiler_image_type::compile_node
( image_type& val, const symbol_table& symbols, const model& the_model,
  const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_image_type );
  CLAW_PRECOND( node.children.size() >= 5 );

  node_compiler_identifier comp_id;
  node_compiler_u_integer_type comp_uint;

  std::string image_name;
  u_integer_type left;
  u_integer_type top;
  u_integer_type width;
  u_integer_type height;

  comp_id.compile_node( image_name, node.children[0] );
  comp_uint.compile_node( left, node.children[1] );
  comp_uint.compile_node( top, node.children[2] );
  comp_uint.compile_node( width, node.children[3] );
  comp_uint.compile_node( height, node.children[4] );

  if ( !the_model.has_image_resource(image_name) )
    error( node.value.begin(), node.value.end(),
           "No image resource has this name." );

  val.set_image_file( the_model.get_image_resource(image_name).get_file() );
  val.set_left( left.get_value() );
  val.set_top( top.get_value() );
  val.set_width( width.get_value() );
  val.set_height( height.get_value() );

  for ( unsigned int i=5; i!=node.children.size(); ++i )
    compile_body_item( val, symbols, node.children[i] );
} // node_compiler_image_type::compile_node()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the body of an image.
 * \param val The current image.
 * \param symbols The symbols of the model.
 * \param node Node to compile.
 */
void mc::node_compiler_image_type::compile_body_item
( image_type& val, const symbol_table& symbols, const tree_node& node ) const
{
  CLAW_PRECOND( node.value.id() == model_grammar::id_image_body_item );
  CLAW_PRECOND( node.children.size() == 2 );

  node_compiler_identifier comp_id;
  std::string field;

  comp_id.compile_node( field, node.children[0] );

  if ( field == "flip_x" )
    set_flip_x( val, symbols, node.children[1] );
  else if ( field == "flip_y" )
    set_flip_y( val, symbols, node.children[1] );
  else if ( field == "alpha" )
    set_alpha( val, symbols, node.children[1] );
  else
    error( node.value.begin(), node.value.end(), "Not a field of image_type." );
} // node_compiler_image_type::compile_body_item()

