/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file monster_item.tpp
 * \brief Implementation of the ptb::monster_item class.
 * \author Sbastien Angibaud
 */
#include <sstream>
#include "engine/base_item.hpp"
#include "ptb/item/stone/stone.hpp"
#include "ptb/monster.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
template<class Base>
ptb::monster_item<Base>::monster_item()
  : m_alpha_blend_injured(1), m_alpha_blend_inc(-0.02)
{
  super::set_can_move_items(false);
} // monster_item::monster_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give a string representation of the item.
 * \param str (out) The result of the convertion.
 */
template<class Base>
void ptb::monster_item<Base>::to_string( std::string& str ) const
{
  super::to_string(str);
  monster::to_string(str);
} // monster_item::to_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief When a collision between two items occurs, hit() is called from the
 *        strongest item.
 * \param that The other item of the collision.
 * \param old_self The state to consider for the current item.
 * \param old_that The state to consider for the item "that".
 */
template<class Base>
void ptb::monster_item<Base>::hit
( bear::engine::base_item& that,
  const bear::universe::physical_item_state& old_self,
  const bear::universe::physical_item_state& old_that )
{
  ptb::monster* other = dynamic_cast<ptb::monster*>(&that);

  if (other != NULL)
    {
      if ( m_offensive_phase )
        other->receive_an_attack(*this);
      else
        super::hit(that, old_self, old_that);
    }
  else
    super::hit(that, old_self, old_that);
} // monster_item::hit()


/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
template<class Base>
void ptb::monster_item<Base>::progress( bear::universe::time_type elapsed_time )
{
  super::progress( elapsed_time );

  if ( m_is_injured )
    {
      bear::time_ref::time_reference current_time;
      current_time.set();

      if ( current_time - m_injured_time >= 1000 )
        finish_injure();

      m_alpha_blend_injured += m_alpha_blend_inc;
      
      if ( m_alpha_blend_injured <= 0.3 )
        m_alpha_blend_inc = 0.02;
      else if ( m_alpha_blend_injured >= 1 )
        m_alpha_blend_inc = -0.02;
      
      super::set_alpha_blend(m_alpha_blend_injured);
    }
  else
    {
      m_alpha_blend_injured = 1;
      m_alpha_blend_inc = -0.02;
      super::set_alpha_blend(1);
    }
} // monster_item::progress()


/*----------------------------------------------------------------------------*/
/**
 * \brief We remove some energy of the item..
 *
 * \param energy The quantity of energy removed.
 */
template<class Base>
void ptb::monster_item<Base>::remove_energy( unsigned int energy )
{
  if ( energy >= m_energy )
    m_energy = 0;
  else
    m_energy -= energy;

  if ( m_energy == 0)
    inform_no_energy();
} // monster_item::remove_energy()




/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the item that he have no energy now.
 */
template<class Base>
void ptb::monster_item<Base>::inform_no_energy()
{
  super::kill();
} // monster_item::inform_no_energy()


