/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file speaker_item.cpp
 * \brief Implementation of the ptb::speaker_item class.
 * \author Sbastien Angibaud
 */
#include "ptb/speaker_item.hpp"
#include "gui/frame.hpp"
#include "gui/static_text.hpp"
#include "engine/camera.hpp"
#include "engine/game.hpp"
#include "engine/font_factory.hpp"
#include "ptb/layer/balloon_layer.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor of a balloon.
 * \param speaker The speaker.
 * \param speech The speech said by the speaker.
 * \param follow Tell if the balloon follows the speaker.
 */
ptb::speaker_item::balloon::balloon
( const std::list< std::string >& speeches  )
  : m_speeches(speeches), m_spike( create_spike() ), m_size_frame(0,0),
    m_has_started(false),
    m_frame( NULL, create_corner(), create_horizontal_border(),
             create_vertical_border(), create_background() )
{
  m_spike->flip( true );

  m_frame.set_size( std::numeric_limits<unsigned int>::max(),
                    std::numeric_limits<unsigned int>::max() );

  m_text = new bear::gui::static_text
    ( &m_frame, bear::engine::font_factory::create("font/speech-10x14.png") );
  
  if ( ! m_speeches.empty() ) 
    write_text();
  
  m_size_frame = m_text->get_size();
  m_frame.set_size( claw::math::coordinate_2d<unsigned int>(0, 0) );  
  m_time = 0;
} // speaker_item::balloon::balloon()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::speaker_item::balloon::~balloon()
{
  delete m_spike;
} // speaker_item::balloon::~balloon()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the players balloon and update local values.
 * \param elapsed_time The time elapsed since the last call.
 */
void
ptb::speaker_item::balloon::progress( bear::universe::time_type elapsed_time )
{
  if ( m_has_started )
    {
      m_time += elapsed_time;
      
      if ( m_time >= m_play_time) 
        {
          if ( m_speeches.empty() ) 
            decrease();
          else
            {
              m_time = 0;
              write_text();
            }
        }
    }
  else
    {
      increase();
      if ( ( m_text->get_size().x == m_size_frame.x ) && 
           ( m_text->get_size().y == m_size_frame.y ) ) 
        m_has_started = true;
    }
} // speaker_item::balloon::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Increase the balloon.
 */
void ptb::speaker_item::balloon::increase()
{
  claw::math::coordinate_2d<unsigned int> size
    ( m_frame.width(), m_frame.height() );

  if ( size.x <= m_size_frame.x - 10 )
    size.x += 10;
  else
    size.x = m_size_frame.x;

  if ( size.y <= m_size_frame.y - 10 )
    size.y += 10;
  else
    size.y = m_size_frame.y;
   
  m_frame.set_size(size);
  m_text->set_size(size);
} // speaker_item::balloon::increase()

/*----------------------------------------------------------------------------*/
/**
 * \brief Decrease the balloon.
 */
void ptb::speaker_item::balloon::decrease()
{
  claw::math::coordinate_2d<unsigned int> size(m_text->get_size());
  
  if ( size.x >= 10 )
    size.x -= 10;
  else
    size.x = 0;
      
  if ( size.y >= 1 )
    size.y -= 1;
  else
    size.y = 0;
  
  m_text->set_size(size);
  m_frame.set_size(size); 
} // speaker_item::balloon::decrease()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render balloon.
 * \param screen The screen on which we will render.
 */
void ptb::speaker_item::balloon::render( bear::visual::screen& screen )
{
  m_frame.render( screen );

  bear::universe::position_type pos(m_frame.get_position());
  
  if ( m_right )
    {
      m_spike->mirror(false);
      pos.x -= m_spike->width();
    }
  else
    {
      m_spike->mirror(true);
      pos.x += m_frame.width();
    }

  if ( !m_top )
    {
      m_spike->flip(true);
      pos.y -= m_spike->height();
    }
  else
    {
      m_spike->flip(false);
      pos.y += m_frame.height();
    }
  
  screen.render( pos.cast_value_type_to<int>(), *m_spike );
} // speaker_item::balloon::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Write the new text.
 */
void ptb::speaker_item::balloon::write_text()
{
  m_frame.set_size( std::numeric_limits<unsigned int>::max(),
                    std::numeric_limits<unsigned int>::max() );

  m_text->set_auto_size(true);
  m_text->set_text(m_speeches.front());
  m_text->set_auto_expand(true);
    
  if ( m_text->width() > 200 ) 
    {
      m_text->set_width(200);      
      m_text->set_text(m_speeches.front());
    }
  
  m_frame.set_size(m_text->get_size());
  
  m_play_time = m_speeches.front().length()/10;

  if ( m_play_time < 2 )
    m_play_time = 2;

  m_speeches.pop_front();
} // speaker_item::balloon::write_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if the balloon is finished.
 */
bool ptb::speaker_item::balloon::is_finished() const
{
  return ( m_has_started && 
           ( m_frame.width() < 10 ) && ( m_frame.height() < 10 ) )  ;
} // speaker_item::balloon::is_finish()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the spike.
 */
bear::visual::sprite* ptb::speaker_item::balloon::create_spike() const
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  const bear::visual::image& img = glob.get_image("gfx/balloon.png");
  claw::math::rectangle<unsigned int> rect(12, 12, 20, 20);

  return new bear::visual::sprite( img, rect );
} // speaker_item::balloon::create_spike()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the corners.
 */
bear::visual::sprite* ptb::speaker_item::balloon::create_corner() const
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  const bear::visual::image& img = glob.get_image("gfx/balloon.png");
  claw::math::rectangle<unsigned int> rect(0, 0, 12, 12);

  return new bear::visual::sprite( img, rect );
} // speaker_item::balloon::create_corner()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the horizontal borders.
 */
bear::visual::sprite*
ptb::speaker_item::balloon::create_horizontal_border() const
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  const bear::visual::image& img = glob.get_image("gfx/balloon.png");
  claw::math::rectangle<unsigned int> rect(12, 0, 20, 12);

  return new bear::visual::sprite( img, rect );
} // speaker_item::balloon::create_horizontal_border()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the vertical borders.
 */
bear::visual::sprite* ptb::speaker_item::balloon::create_vertical_border() const
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  const bear::visual::image& img = glob.get_image("gfx/balloon.png");
  claw::math::rectangle<unsigned int> rect(0, 12, 12, 20);

  return new bear::visual::sprite( img, rect );
} // speaker_item::balloon::create_vertical_border()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the background.
 */
bear::visual::sprite* ptb::speaker_item::balloon::create_background() const
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  const bear::visual::image& img = glob.get_image("gfx/balloon.png");
  claw::math::rectangle<unsigned int> rect(12, 3, 20, 9);

  return new bear::visual::sprite( img, rect );
} // speaker_item::balloon::create_background()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the frame.
 */
bear::gui::frame* ptb::speaker_item::balloon::get_frame()
{
  return &m_frame;
} // balloon::get_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the balloon.
 */
void ptb::speaker_item::balloon::set_position
(const claw::math::coordinate_2d<unsigned int>& pos)
{
  m_frame.set_position(pos);
} // balloon::set_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the right status (for the spike).
 */
void ptb::speaker_item::balloon::set_spike_right(bool right)
{
  m_right = right;
} // balloon::set_spike_right()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the top status (for the spike).
 */
void ptb::speaker_item::balloon::set_spike_top(bool top)
{
  m_top = top;
} // balloon::set_spike_top()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::speaker_item::speaker_item()
  : m_balloon(NULL), m_speak_state(false)
{

} // speaker_item::speaker_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::speaker_item::~speaker_item()
{
  if ( m_balloon != NULL )
    delete m_balloon;
} // speaker_item::~speaker_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Indicates if the speaker has finished to chat.
 */
bool ptb::speaker_item::has_finished_to_chat() const
{
  if ( m_balloon != NULL )
    return m_balloon->is_finished();
  else
    return true;
} // speaker_item::has_finished_to_chat()

/*----------------------------------------------------------------------------*/
/**
 * \brief Stop to speak.
 */
void ptb::speaker_item::stop_to_speak()
{
  if ( m_balloon != NULL )
    delete m_balloon;

  m_balloon = NULL;
} // speaker_item::stop_to_speak()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start the speech.
 */
bool ptb::speaker_item::speak()
{
  bool ok = true;

  balloon_layer::msg_add_speaker msg_speaker;
  msg_speaker.set_speaker(this);
  
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  ok =  glob.send_message( balloon_layer::default_name(), msg_speaker );

  return ok;
} // speaker_item::speak()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create a new balloon.
 */
void ptb::speaker_item::create_balloon()
{
  if ( m_balloon != NULL )
    delete m_balloon;

  m_balloon = new balloon(m_speeches);
} // speaker_item::create_balloon()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return a pointer on the speaker.
 */
const bear::universe::physical_item_state*
ptb::speaker_item::get_speaker() const
{
  return m_speaker;
} // speaker_item::get_speaker()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a speech.
 */
void ptb::speaker_item::add_speech( const std::string& speech )
{
  m_speeches.push_back(speech);
} // speaker_item::add_speech()

/*----------------------------------------------------------------------------*/
/**
 * \brief Reset the speeches.
 */
void ptb::speaker_item::reset_speeches()
{
  m_speeches.clear();
} // speaker_item::reset_speeches()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the balloon.
 */
ptb::speaker_item::balloon* ptb::speaker_item::get_balloon() const
{
  return m_balloon;
} // speaker_item::get_balloon()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the speaker item.
 *
 * param speaker_item A pointer on the speaker item.
 */
void ptb::speaker_item::set_speaker_item
(const bear::universe::physical_item_state* speaker)
{
  m_speaker = speaker;
} // speaker_item::set_speaker_item()
