/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file frame.cpp
 * \brief Implementation of the ptb::frame class.
 * \author Julien Jorge
 */
#include "ptb/frame/frame.hpp"

#include "engine/game.hpp"
#include "ptb/layer/windows_layer.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owning_layer The layer onwing the window.
 */
ptb::frame::frame( windows_layer* owning_layer )
  : bear::gui::window( NULL, create_frame() ), m_owning_layer(owning_layer)
{
  CLAW_PRECOND(owning_layer != NULL);
  set_size( bear::engine::game::get_instance().get_screen_size() );
} // frame::frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called when the frame is displayed.
 */
void ptb::frame::on_focus()
{
  // nothing to do
} // frame::on_focus()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite to use for the cursor.
 */
bear::visual::sprite* ptb::frame::get_cursor()
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  const bear::visual::image& img = glob.get_image( "gfx/frame.tga" );
  claw::math::rectangle<unsigned int> rect( 40, 3, 24, 29 );

  return new bear::visual::sprite( img, rect );
} // frame::get_cursor()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show a window, centered in the scrren, to replace this window.
 * \param wnd The class name of the window to show.
 */
void ptb::frame::show_window( frame* wnd ) const
{
  CLAW_PRECOND(wnd != NULL);
  m_owning_layer->show_centered_window(wnd);
} // frame::show_window()

/*----------------------------------------------------------------------------*/
/**
 * \brief Close the current window (come back to the previous one).
 */
void ptb::frame::close_window() const
{
  m_owning_layer->close_window();
} // frame::close_window()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the frame in the window.
 */
bear::gui::frame& ptb::frame::get_frame()
{
  return *m_content_frame;
} // frame::get_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the frame contained in the window.
 */
bear::gui::frame* ptb::frame::create_frame()
{
  m_content_frame = new bear::gui::frame
    ( NULL, create_corner(), create_horizontal_border(),
      create_vertical_border(), create_background() );

  m_content_frame->set_background_alpha( 0.75 );

  return m_content_frame;
} // frame::create_corner()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the corners.
 */
bear::visual::sprite* ptb::frame::create_corner() const
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  const bear::visual::image& img = glob.get_image( "gfx/frame.tga" );
  claw::math::rectangle<unsigned int> rect(0, 0, 32, 32);

  return new bear::visual::sprite( img, rect );
} // frame::create_corner()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the horizontal borders.
 */
bear::visual::sprite* ptb::frame::create_horizontal_border() const
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  const bear::visual::image& img = glob.get_image( "gfx/frame.tga" );
  claw::math::rectangle<unsigned int> rect(0, 32, 32, 8);

  return new bear::visual::sprite( img, rect );
} // frame::create_horizontal_border()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the vertical borders.
 */
bear::visual::sprite* ptb::frame::create_vertical_border() const
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  const bear::visual::image& img = glob.get_image( "gfx/frame.tga" );
  claw::math::rectangle<unsigned int> rect(32, 0, 8, 32);

  return new bear::visual::sprite( img, rect );
} // frame::create_vertical_border()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the background.
 */
bear::visual::sprite* ptb::frame::create_background() const
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  const bear::visual::image& img = glob.get_image( "gfx/frame.tga" );
  claw::math::rectangle<unsigned int> rect(32, 32, 32, 32);

  return new bear::visual::sprite( img, rect );
} // frame::create_background()
