/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file message_box.cpp
 * \brief Implementation of the ptb::message_box class.
 * \author Julien Jorge
 */
#include "ptb/frame/message_box.hpp"

#include <claw/string_algorithm.hpp>
#include "engine/font_factory.hpp"
#include "engine/string_base.hpp"
#include "text/text_metric.hpp"

/*----------------------------------------------------------------------------*/
const ptb::message_box::flags ptb::message_box::s_ok     = (1 << 0);
const ptb::message_box::flags ptb::message_box::s_cancel = (1 << 1);

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param in_layer The layer containing this frame.
 * \param msg_key The key (in the string_base) of the text to display.
 * \param buttons (in) The buttons to display. (out) The selected button.
 *        Can be NULL if you don't care about the result.
 *
 * A button labelled 'Ok' is created, even if the corresponding flag is not set.
 */
ptb::message_box::message_box
( windows_layer* in_layer, const std::string& msg_key, flags* buttons )
  : frame(in_layer), m_flags(buttons), m_cursor_position(0), m_margin(10)
{
  create_controls(msg_key);

  if ( m_flags != NULL )
    *m_flags = 0;

  position_cursor();
} // message_box::message_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a key has been pressed.
 * \param key The code of the key.
 */
bool ptb::message_box::on_key_press( bear::input::keyboard::key_code key )
{
  bool result = true;

  switch( key )
    {
    case bear::input::keyboard::kc_new_line:
    case bear::input::keyboard::kc_keypad_enter:
      result = m_button[m_cursor_position]->on_mouse_press
        ( bear::input::mouse::mc_left_button,
          claw::math::coordinate_2d<unsigned int>(0, 0) );
      break;
    case bear::input::keyboard::kc_escape:
      result = on_cancel();
      break;
    case bear::input::keyboard::kc_left:
      result = on_left();
      break;
    case bear::input::keyboard::kc_right:
      result = on_right();
      break;
    default:
      result = false;
    }

  return result;
} // message_box::on_key_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a joystick button has been pressed.
 * \param button The code of the button.
 * \param joy_index The index of the joytick.
 */
bool ptb::message_box::on_button_press
( bear::input::joystick::joy_code button, unsigned int joy_index )
{
  bool result = true;

  switch( button )
    {
    case bear::input::joystick::jc_axis_left:
      result = on_left();
      break;
    case bear::input::joystick::jc_axis_right:
      result = on_right();
      break;
    case bear::input::joystick::jc_button_1:
    case bear::input::joystick::jc_button_2:
    case bear::input::joystick::jc_button_3:
    case bear::input::joystick::jc_button_4:
    case bear::input::joystick::jc_button_5:
    case bear::input::joystick::jc_button_6:
    case bear::input::joystick::jc_button_7:
    case bear::input::joystick::jc_button_8:
    case bear::input::joystick::jc_button_9:
    case bear::input::joystick::jc_button_10:
    case bear::input::joystick::jc_button_11:
    case bear::input::joystick::jc_button_12:
    case bear::input::joystick::jc_button_13:
    case bear::input::joystick::jc_button_14:
    case bear::input::joystick::jc_button_15:
    case bear::input::joystick::jc_button_16:
      result = m_button[m_cursor_position]->on_mouse_press
        ( bear::input::mouse::mc_left_button,
          claw::math::coordinate_2d<unsigned int>(0, 0) );
      break;
    default:
      result = false;
    }

  return result;
} // message_box::on_button_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called when a mouse moves over the frame.
 * \param pos The position of the mouse.
 */
bool ptb::message_box::on_mouse_move
( const claw::math::coordinate_2d<unsigned int>& pos )
{
  bool result = false;

  for ( unsigned int i=0; !result && i!=m_button.size(); ++i )
    if ( m_button[i]->get_rectangle().includes(pos) )
      {
        m_cursor_position = i;
        result = true;
      }

  if ( result )
    position_cursor();

  return result;
} // message_box::on_mouse_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on 'ok'.
 */
bool ptb::message_box::on_ok()
{
  if ( m_flags != NULL )
    *m_flags |= s_ok;

  close_window();

  return true;
} // message_box::on_ok()

/*----------------------------------------------------------------------------*/
/**
 * \brief Answer to a click on 'cancel'.
 */
bool ptb::message_box::on_cancel()
{
  if ( m_flags != NULL )
    *m_flags |= s_cancel;

  close_window();

  return true;
} // message_box::on_cancel()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the cursor to the left.
 */
bool ptb::message_box::on_left()
{
  if ( m_cursor_position == 0 )
    m_cursor_position = m_button.size();

  --m_cursor_position;
  position_cursor();

  return true;
} // message_box::on_left()

/*----------------------------------------------------------------------------*/
/**
 * \brief Move the cursor to the right.
 */
bool ptb::message_box::on_right()
{
  m_cursor_position = (m_cursor_position + 1) % m_button.size();
  position_cursor();

  return true;
} // message_box::on_right()

/*----------------------------------------------------------------------------*/
/**
 * \brief Position the cursor near the selected item.
 */
void ptb::message_box::position_cursor()
{
  button_with_text* btn = m_button[m_cursor_position];

  unsigned int x;
  unsigned int y;

  x = btn->get_position().x - m_cursor->width() - m_margin;
  y = btn->get_position().y + (int)(btn->height() - m_cursor->height()) / 2;

  m_cursor->set_position(x, y);
} // message_box::position_cursor()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls in this frame.
 * \param msg_key The name of the string to display.
 */
void ptb::message_box::create_controls( const std::string& msg_key )
{
  m_cursor = new bear::gui::picture( this, frame::get_cursor() );

  bear::gui::static_text::font_type font =
    bear::engine::font_factory::create("font/fixed_yellow-10x20.png");

  create_text( msg_key, font );
  create_buttons( font );

  const button_with_text* btn = m_button.back();
  const unsigned int r = std::max( btn->right(), m_text->right() );

  set_size( r + m_margin, btn->bottom() + m_margin );
} // message_box::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls that displays the text.
 * \param msg_key The name of the string to display.
 * \param f The font to use for the texts.
 */
void ptb::message_box::create_text
( const std::string& msg_key, bear::gui::static_text::font_type f )
{
  m_text = new bear::gui::static_text( this, f );
  m_text->set_position( m_margin, m_margin );

  m_text->set_size( get_size() / 2 );
  std::string text(msg_key);
  bear::engine::string_base::get_instance().get_string(text);
  m_text->set_text(text);
  m_text->set_auto_expand( true );
} // message_box::create_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the buttons.
 * \param f The font to use for the texts.
 */
void ptb::message_box::create_buttons( bear::gui::static_text::font_type f )
{
  button_with_text* btn =
    new button_with_text( *this, std::mem_fun(&message_box::on_ok), this );

  std::string text("ok");
  btn->set_font( f );
  btn->set_auto_size( true );
  bear::engine::string_base::get_instance().get_string(text);
  btn->set_text(text);

  m_button.push_back( btn );

  if ( m_flags != NULL )
    if ( *m_flags & s_cancel )
      {
        btn = new button_with_text
          ( *this, std::mem_fun(&message_box::on_cancel), this );

        btn->set_font( f );
        btn->set_auto_size( true );
        text = "cancel";
        bear::engine::string_base::get_instance().get_string(text);
        btn->set_text(text);

        m_button.push_back( btn );
      }

  position_buttons();
} // message_box::create_buttons()

/*----------------------------------------------------------------------------*/
/**
 * \brief Position the buttons.
 * \param f The font to use for the texts.
 */
void ptb::message_box::position_buttons()
{
  unsigned int button_w = 0;

  for (unsigned int i=0; i!=m_button.size(); ++i)
    button_w += m_cursor->width() + m_margin + m_button[i]->width();

  button_w += (m_button.size() - 1) * m_margin;

  unsigned int frame_w = button_w;

  if ( frame_w < m_text->width() )
    frame_w = m_text->width();

  unsigned int x = (frame_w - button_w) / 2 + m_margin;
  const unsigned int y = m_text->bottom() + m_margin;

  unsigned int i=0;
  m_button[i]->set_position( x + m_cursor->width() + m_margin, y );

  for ( ++i; i!=m_button.size(); ++i )
    m_button[i]->set_position
      ( m_button[i-1]->right() + m_margin + m_cursor->width() + m_margin, y );
} // message_box::position_buttons()
