/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file plee.cpp
 * \brief Implementation of the ptb::plee class.
 * \author Julien Jorge
 */
#include "ptb/item/plee/plee.hpp"

#include <sstream>
#include "engine/camera.hpp"
#include "engine/game.hpp"
#include "engine/world.hpp"
#include "universe/collision_event/collision_event.hpp"
#include "ptb/monster.hpp"
#include "ptb/item/stone/stone.hpp"
#include "ptb/item/plee/state_walk.hpp"
#include "ptb/item/plee/state_idle.hpp"
#include "ptb/item/plee/state_jump.hpp"
#include "ptb/item/plee/state_fall.hpp"
#include "ptb/item/plee/state_dead.hpp"
#include "ptb/item/plee/state_run.hpp"
#include "ptb/item/plee/state_slap.hpp"
#include "ptb/item/plee/state_start_jump.hpp"
#include "ptb/item/plee/state_vertical_jump.hpp"
#include "ptb/item/plee/state_look_up.hpp"
#include "ptb/item/plee/state_crouch.hpp"
#include "ptb/item/plee/state_captive.hpp"
#include "ptb/item/plee/state_throw.hpp"
#include "ptb/item/plee/state_start_throw.hpp"
#include "ptb/item/plee/state_wait.hpp"

/*----------------------------------------------------------------------------*/
const unsigned int ptb::plee::s_max_energy = 100;
const unsigned int ptb::plee::s_states_cardinality = 15;
const bear::universe::time_type ptb::plee::s_time_to_crouch = 0.5;
const bear::universe::time_type ptb::plee::s_time_to_look_up = 0.5;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param name The name of the player.
 */
ptb::plee::plee()
  : m_current_state(idle_state), m_positive_orientation(true), 
    m_stones_count(200), m_status_look_up(false), m_status_crouch(false)
{
  set_speaker_item(this);
  set_mass(1);
  set_size( 50, 110 );
  set_speed_epsilon( bear::universe::speed_type(0.6, 0.6) );
  set_animation_alignment( align_center_on_bottom );

  m_offensive_force = 10;
  m_energy = 100;
  m_monster_type = monster::player_monster;
  m_offensive_phase = false;
  m_offensive_coefficients[normal_attack] = 1;
  
  save_position();
 
  m_states.resize(s_states_cardinality);
  m_states[walk_state] = new state_walk(this);
  m_states[idle_state] = new state_idle(this);
  m_states[idle_state] = new state_idle(this);
  m_states[jump_state] = new state_jump(this);
  m_states[fall_state] = new state_fall(this);
  m_states[dead_state] = new state_dead(this);
  m_states[run_state] = new state_run(this);
  m_states[slap_state] = new state_slap(this);
  m_states[start_jump_state] = new state_start_jump(this);
  m_states[vertical_jump_state] = new state_vertical_jump(this);
  m_states[look_up_state] = new state_look_up(this);
  m_states[crouch_state] = new state_crouch(this);
  m_states[captive_state] = new state_captive(this);
  m_states[start_throw_state] = new state_start_throw(this);
  m_states[throw_state] = new state_throw(this);
  m_states[wait_state] = new state_wait(this);
} // plee::plee()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
ptb::plee::~plee()
{
  unsigned int i;

  for (i=0; i!=s_states_cardinality; ++i)
    delete m_states[i];
} // plee::~plee()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress( bear::universe::time_type elapsed_time )
{
  if ( is_crushed() )
    set_state(plee::dead_state);
  else
    {
      m_states[m_current_state]->progress( elapsed_time );
      progress_spot( elapsed_time );
      
      if ( get_speed().x < 0 )
        m_positive_orientation = false;
      else if ( get_speed().x > 0 )
        m_positive_orientation = true;
    }
} // plee::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite representing the item.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void
ptb::plee::get_visual( std::list<bear::engine::scene_visual>& visuals ) const
{
  super::get_visual(visuals);

  m_states[m_current_state]->get_visual(visuals);
} // plee::get_visual()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do post creation actions.
 */
void ptb::plee::start()
{
  super::start();
  save_position();

  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  glob.get_sound("sound/grr.wav").play();

  set_state(idle_state);
  m_positive_orientation = true;
  
  bear::visual::animation* halo_anim =
    new bear::visual::animation( get_animation("halo") );
  m_states[start_jump_state]->set_animation("halo", halo_anim);
} // plee::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the position of the plee.
 */
void ptb::plee::save_position()
{
  m_saved_position = get_position();
} // plee::save_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell to Plee to start an action.
 * \param a The action to start.
 */
void ptb::plee::start_action( action a )
{
  switch( a )
    {
    case player_action::idle : break;
    case player_action::move_left : break;
    case player_action::move_right : break;
    case player_action::jump : 
      m_states[m_current_state]->do_jump(); break;
    case player_action::get_camera : do_get_camera(); break;
    case player_action::slap : 
      m_states[m_current_state]->do_slap(); break;
    case player_action::look_up : do_start_look_up(); break;
    case player_action::crouch :  do_start_crouch(); break;
    case player_action::drop : break;
    case player_action::chain : break;
    case player_action::unchain : break;
    case player_action::throw_stone :
      m_states[m_current_state]->do_start_throw(); break;
    case action_die : break;
    case action_null: break;
    default:
      super::do_action(a);
    }
} // plee::start_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell to the plee to do an action.
 * \param a The action to do.
 */
void ptb::plee::do_action( action a )
{
  switch( a )
    {
    case player_action::idle : set_state(idle_state); break;
    case player_action::move_left : 
      m_states[m_current_state]->do_move_left(); break;
    case player_action::move_right : 
      m_states[m_current_state]->do_move_right(); break;
    case player_action::jump : break;
    case player_action::get_camera : break;
    case player_action::slap : break;
    case player_action::look_up :  
      m_states[m_current_state]->do_continue_look_up(); break;
    case player_action::crouch : 
      m_states[m_current_state]->do_continue_crouch(); break;
    case player_action::drop : break;
    case player_action::chain : 
      m_states[m_current_state]->chain(); break;
    case player_action::unchain : 
      m_states[m_current_state]->unchain(); break;
    case player_action::throw_stone : break;
    case action_die : do_die(); break;
    case action_null: break;
    default:
      super::do_action(a);
    }
} // plee::do_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell to plee to stop an action.
 * \param a The action to stop.
 */
void ptb::plee::stop_action( action a )
{
  switch( a )
    {
    case player_action::idle : break;
    case player_action::move_left : break;
    case player_action::move_right : break;
    case player_action::jump : 
      m_states[m_current_state]->do_stop_vertical_jump(); break;
    case player_action::get_camera : break;
    case player_action::slap : break;
    case player_action::look_up : do_stop_look_up(); break;
    case player_action::crouch : 
      do_stop_crouch(); break;
    case player_action::throw_stone :
      m_states[m_current_state]->do_stop_throw(); break;
    case player_action::drop : break;
    case player_action::chain : break;
    case player_action::unchain : break;
    case action_die :  break;
    case action_null: break;
    default:
      super::do_action(a);
    }
} // plee::stop_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the state of Plee.
 */
void ptb::plee::set_state(plee_state_name state)
{
  m_states[m_current_state]->stop();
  m_current_state = state;
  m_states[m_current_state]->start();
  reset_animation();
} // plee::set_state()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call the execution of action to the parent in order to
 *  start the animation.
 */
void ptb::plee::start_action_parent(const std::string action)
{
  super::execute_action(action);

  if ( m_positive_orientation )
    mirror( false );
  else
    mirror( true );
} // plee::start_action_parent()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if Plee is crushed.
 */
bool ptb::plee::is_crushed() const
{
  return ( ( ( has_bottom_contact() && has_top_contact() ) ||
             ( has_right_contact() && has_left_contact() ) )  &&
           has_contact_after_collision() );
} // plee::is_crushed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the ratio of the length of preparation of the jump.
 * \param r The length of preparation of the jump.
 */
void ptb::plee::set_jump_time_ratio( double r )
{
  m_jump_time_ratio = r;
} // plee::set_jump_time_ratio()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give the ratio of the length of preparation of the jump.
 */
bear::universe::time_type ptb::plee::get_jump_time_ratio() const
{
  return m_jump_time_ratio;
} // plee::get_jump_time_ratio()

/*----------------------------------------------------------------------------*/
/**
 * \brief Spawn the plee after his death.
 * \todo Game over if the plee has no more lifes.
 * \todo Update lifes and any other counters.
 */
void ptb::plee::regenerate()
{
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  glob.get_sound("sound/grr.wav").play();

  set_position( m_saved_position );
  set_speed( bear::universe::speed_type(0, 0) );
  set_acceleration( bear::universe::force_type(0, 0) );
  m_energy = 100;
  finish_injure();
  set_environment_friction(1);
  set_state(idle_state);
  m_positive_orientation = true;
  stop_to_speak();
} // plee::regenerate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create a stone.
 *
 * \param a The force applied to thes stone.
 */
void ptb::plee::create_stone( const bear::universe::force_type& a )
{
  if ( m_stones_count > 0 )
    {
      --m_stones_count;

      bear::engine::world* w = get_owner();

      stone* new_stone = new stone(get_id());

      new_stone->set_real_field("pos_x", get_center_of_mass().x );
      new_stone->set_real_field("pos_y", get_center_of_mass().y );
      new_stone->set_monster_creator(this);

      w->register_item( new_stone );

      new_stone->start();

      new_stone->add_force(a);
      new_stone->set_environment_friction(get_environment_friction());
    }
} // ptb::plee::create_stone()

/*----------------------------------------------------------------------------*/
/**
 * \brief Plee receives energy.
 *
 * \param energy The energy received.
 */
void ptb::plee::receive_energy(unsigned int energy)
{
  m_energy += energy;

  if ( m_energy > s_max_energy )
    m_energy = s_max_energy;

} // plee::receive_energy()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give the number of stones.
 */
unsigned int ptb::plee::get_stones_count() const 
{
  return m_stones_count;
} // plee::get_stones_count()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if Plee has a positive orientaion.
 */
bool ptb::plee::has_positive_orientation() const
{
  return m_positive_orientation;
} // plee::has_positive_orientation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the look up status.
 */
void ptb::plee::set_status_look_up(bool status)
{
  m_status_look_up = status;
} // plee::set_status_look_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give the look up status.
 */
bool ptb::plee::get_status_look_up() const
{
  return m_status_look_up;
} // plee::get_status_look_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the crouch status.
 */
void ptb::plee::set_status_crouch(bool status)
{
  m_status_crouch = status;
} // plee::set_status_look_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give the crouch status.
 */
bool ptb::plee::get_status_crouch() const
{
  return m_status_crouch;
} // plee::get_status_crouch()

/*----------------------------------------------------------------------------*/
/**
 * \brief Make the plee start to look up.
 */
void ptb::plee::do_start_look_up()
{
  m_status_look_up = true;
  m_look_up_time = 0;
  m_states[m_current_state]->do_look_up(); 
} // plee::do_start_look_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Make the plee start to crouch.
 */
void ptb::plee::do_start_crouch()
{
  m_status_crouch = true;
  m_crouch_time = 0;
  m_states[m_current_state]->do_crouch(); 
} // plee::do_start_crouch()

/*----------------------------------------------------------------------------*/
/**
 * \brief  Inform the item that he have no energy now.
 */
void ptb::plee::inform_no_energy()
{
  do_die();
} // plee::inform_no_energy()

/*----------------------------------------------------------------------------*/
/**
 * \brief Give a string representation of the item.
 * \param str (out) The result of the convertion.
 */
void ptb::plee::to_string( std::string& str ) const
{
  std::ostringstream oss;

  super::to_string(str);

  oss << "state: ";
  oss << m_states[m_current_state]->get_name();
  oss << "\n";
  oss << "status_look_up: " << m_status_look_up << "\n";
  oss << "status_crouch: " << m_status_crouch << "\n";

  str += oss.str();
} // plee::to_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the camera on the current plee.
 */
void ptb::plee::do_get_camera()
{
  bear::engine::camera::placement_mode mode;

  if ( get_name() == bear::engine::player::player_name(1) )
    mode = bear::engine::camera::lock_first_player;
  else
    mode = bear::engine::camera::lock_second_player;

  bear::engine::camera::set_placement_message msg(mode);
  bear::engine::level_globals& glob =
    bear::engine::game::get_instance().current_level_globals();

  glob.send_message( bear::engine::camera::default_name(), msg );
} // plee::do_get_camera()

/*----------------------------------------------------------------------------*/
/**
 * \brief Die.
 * \todo We should start a timer to definitely kill the plee in few seconds.
 * \todo The camera should not follow the plee when he's dying. More over
 */
void ptb::plee::do_die()
{
  set_state(dead_state);
} // plee::do_die()

/*----------------------------------------------------------------------------*/
/**
 * \brief Stop to look up.
 */
void ptb::plee::do_stop_look_up()
{
  m_status_look_up = false;
  m_states[m_current_state]->do_stop_look_up(); 
} // state_plee::do_stop_look_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Stop to crouch.
 */
void ptb::plee::do_stop_crouch()
{
  m_status_crouch = false;
  m_states[m_current_state]->do_stop_crouch(); 
} // state_plee::do_stop_crouch()

/*----------------------------------------------------------------------------*/
/**
 * \brief Progress the spot.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::plee::progress_spot( bear::universe::time_type elapsed_time )
{
  bool balance = true;

  if ( m_status_look_up ) 
    {
      m_look_up_time += elapsed_time;
      
      if ( m_look_up_time >= s_time_to_look_up )
        {
          balance = false;
          claw::math::coordinate_2d<int> gap(0,-5);
          add_spot_gap(gap);
        }
    }

  if ( m_status_crouch ) 
    {
      m_crouch_time += elapsed_time;

      if ( m_crouch_time >= s_time_to_crouch )
        {
          balance = false;
          claw::math::coordinate_2d<int> gap(0,5);
          add_spot_gap(gap);
        }
    }

  if ( balance ) 
    balance_spot();
} // plee::progress_spot()
