/*
  Plee The Bear

  Copyright (C) 2005-2008 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file state_wasp_fly.cpp
 * \brief Implementation of the ptb::state_wasp_fly class.
 * \author Sebastien Angibaud
 */
#include "ptb/item/wasp/state_wasp_fly.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param wasp_instance The pointer on the wasp.
 */
ptb::state_wasp_fly::state_wasp_fly( ptb::wasp* wasp_instance )
  : state_wasp(wasp_instance),
    m_come_back(false), m_oscillation_angle(0), m_current_step(0),
    m_have_attacked(false)
{

} // state_wasp_fly::state_wasp_fly()

/*----------------------------------------------------------------------------*/
/**
 * \brief Return the name of the state.
 */
std::string ptb::state_wasp_fly::get_name() const
{
  return "fly";
} // state_wasp_fly::get_name()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::state_wasp_fly::progress( bear::universe::time_type elapsed_time )
{
  if( ( !m_have_attacked ) && search_player() )
    {
      // we start an offensive
      m_have_attacked = true;
      m_wasp_instance->set_state(ptb::wasp::attack_state);
    }
  else
    {
      bear::universe::position_type target1 =
        m_wasp_instance->get_first_target();
      bear::universe::position_type target2 =
        m_wasp_instance->get_second_target();
      
      if ( m_come_back )
        next_position(target2, target1);
      else
        next_position(target1, target2);
        
      if( m_current_step == m_wasp_instance->get_number_of_steps())
        {
            m_come_back = !m_come_back;            
            m_current_step = 0;
            m_have_attacked = false;
            oriente();
        }
    }
} // state_wasp_fly::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialization of this state.
 */
void ptb::state_wasp_fly::start()
{
  oriente();

  m_first_player =
    bear::engine::pointer_to_player( bear::engine::player::player_name(1) );
  m_second_player =
    bear::engine::pointer_to_player( bear::engine::player::player_name(2) );

  m_wasp_instance->start_action_parent("fly");
} // state_wasp_fly::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Oriente the wasp.
 */
void ptb::state_wasp_fly::oriente()
{
 if ( m_come_back )
   if ( m_wasp_instance->get_first_target().x <
        m_wasp_instance->get_second_target().x )
     m_wasp_instance->set_positive_orientation(false);
   else
     m_wasp_instance->set_positive_orientation(true);
 else
   if ( m_wasp_instance->get_first_target().x <
        m_wasp_instance->get_second_target().x )
     m_wasp_instance->set_positive_orientation(true);
   else
     m_wasp_instance->set_positive_orientation(false);
} // state_wasp_fly::oriente()


/*---------------------------------------------------------------------------*/
/**
 * \brief Go to the next position.
 *  \param origin The origin position.
 *  \param target The target position.
 */
void ptb::state_wasp_fly::next_position
( bear::universe::position_type& origin,
  bear::universe::position_type& target)
{
  bear::universe::position_type position;

  m_current_step++;

  position.x = origin.x +
    ( ( target.x - origin.x ) * m_current_step )
    /  m_wasp_instance->get_number_of_steps();

  position.y = origin.y +
    ( ( target.y - origin.y ) * m_current_step )
    /  m_wasp_instance->get_number_of_steps();

  m_oscillation_angle += ( 3.14 / 12 );
  if ( m_oscillation_angle >= 6.28 )
    m_oscillation_angle -= 6.28;

  position.y += ( (sin(m_oscillation_angle)) * 15);

  m_wasp_instance->set_position( position  + m_wasp_instance->get_position()
                                 - m_wasp_instance->get_center_of_mass() );
} // state_wasp_fly::next_position()




/*---------------------------------------------------------------------------*/
/**
 * \brief Test if there is a player in the accesive zone.
 */
bool ptb::state_wasp_fly::search_player() const
{
  bool result = false;

  if(m_first_player)
    result = player_in_zone(m_first_player->get_center_of_mass());

  if( (! result) && (m_second_player))
    result = player_in_zone(m_second_player->get_center_of_mass());

  return result;
} // state_wasp_fly::search_player()


/*---------------------------------------------------------------------------*/
/**
 * \brief Test if the player is in the accesive zone.
 * \param pos Position of the player.
 */
bool ptb::state_wasp_fly::player_in_zone
( const bear::universe::position_type& pos ) const
{
  bear::universe::speed_type vect(m_wasp_instance->get_center_of_mass(), pos);

  if ( m_wasp_instance->get_positive_orientation() )
    if ( ( vect.x > 0 ) && ( vect.x < ( 3 * vect.y ) ) &&
       vect.length() < 400 )
      return true;
    else
      return false;
  else
    if ( ( vect.x > ( -3 * vect.y ) ) && ( vect.x < 0 ) &&
       vect.length() < 400 )
      return true;
    else
      return false;
} // state_wasp_fly::player_in_zone()


